<?php
// accounts.php - Manage IMAP/Exchange Accounts (Fixed - Now Adds Accounts Correctly)
require_once 'config.php';
require_once 'includes/db.php';
require_once 'includes/functions.php';
require_once 'includes/header.php';

$message = '';

// Provider presets
$presets = [
    'gmail' => ['host' => 'imap.gmail.com', 'port' => 993, 'encryption' => 'ssl'],
    'outlook' => ['host' => 'outlook.office365.com', 'port' => 993, 'encryption' => 'ssl'],
    'yahoo' => ['host' => 'imap.mail.yahoo.com', 'port' => 993, 'encryption' => 'ssl'],
    'aol' => ['host' => 'imap.aol.com', 'port' => 993, 'encryption' => 'ssl'],
    'zoho' => ['host' => 'imap.zoho.com', 'port' => 993, 'encryption' => 'ssl'],
    'rackspace' => ['host' => 'secure.emailsrvr.com', 'port' => 993, 'encryption' => 'ssl'],
    'smarsh' => ['host' => 'mail.smarshmail.com', 'port' => 993, 'encryption' => 'ssl'],
    'ionos_com' => ['host' => 'imap.ionos.com', 'port' => 993, 'encryption' => 'ssl'],
    'ionos_de' => ['host' => 'imap.ionos.de', 'port' => 993, 'encryption' => 'ssl'],
    'ionos_co_uk' => ['host' => 'imap.ionos.co.uk', 'port' => 993, 'encryption' => 'ssl'],
    'ionos_fr' => ['host' => 'imap.ionos.fr', 'port' => 993, 'encryption' => 'ssl'],
];

// Handle form submission (add, edit, toggle, delete)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if ($action === 'add' || $action === 'edit') {
        $id = (int)($_POST['id'] ?? 0);
        $provider = $_POST['provider'] ?? 'custom';
        $host = trim($_POST['host'] ?? '');
        $port = (int)($_POST['port'] ?? 993);
        $encryption = $_POST['encryption'] ?? 'ssl';
        $username = trim($_POST['username'] ?? '');
        $password = $_POST['password'] ?? '';
        $forward_all = isset($_POST['forward_all']);
        $keywords = trim($_POST['keywords'] ?? '');

        if (empty($host) || empty($username) || empty($password)) {
            $message = '<div class="alert alert-danger"><i class="bi bi-x-circle"></i> All fields are required.</div>';
        } else {
            $encrypted_pass = encrypt_data($password);
            $rules = $forward_all 
                ? json_encode(['forward_all' => true]) 
                : json_encode(['keywords' => array_filter(array_map('trim', explode("\n", $keywords)))]);

            try {
                if ($action === 'add') {
                    $stmt = $pdo->prepare("INSERT INTO accounts (provider, host, port, encryption, username, password_encrypted, rules) VALUES (?, ?, ?, ?, ?, ?, ?)");
                    $stmt->execute([$provider, $host, $port, $encryption, $username, $encrypted_pass, $rules]);
                    $message = '<div class="alert alert-success"><i class="bi bi-check-circle"></i> Account added successfully!</div>';
                    log_action('info', "Account added: $username");
                } elseif ($action === 'edit' && $id > 0) {
                    $stmt = $pdo->prepare("UPDATE accounts SET provider = ?, host = ?, port = ?, encryption = ?, username = ?, password_encrypted = ?, rules = ? WHERE id = ?");
                    $stmt->execute([$provider, $host, $port, $encryption, $username, $encrypted_pass, $rules, $id]);
                    $message = '<div class="alert alert-success"><i class="bi bi-check-circle"></i> Account updated successfully!</div>';
                    log_action('info', "Account updated: ID $id");
                }
            } catch (Exception $e) {
                $message = '<div class="alert alert-danger"><i class="bi bi-exclamation-triangle"></i> Database error: ' . htmlspecialchars($e->getMessage()) . '</div>';
            }
        }
    } elseif ($action === 'toggle') {
        $id = (int)$_POST['id'];
        $active = (int)($_POST['active'] ?? 0);
        $stmt = $pdo->prepare("UPDATE accounts SET active = ? WHERE id = ?");
        $stmt->execute([$active ? 0 : 1, $id]);
        $message = '<div class="alert alert-success">Account status updated.</div>';
    } elseif ($action === 'delete') {
        $id = (int)$_POST['id'];
        $stmt = $pdo->prepare("DELETE FROM accounts WHERE id = ?");
        $stmt->execute([$id]);
        $message = '<div class="alert alert-success">Account deleted.</div>';
        log_action('info', "Account deleted: ID $id");
    }
}

// Fetch all accounts for display
$accounts = $pdo->query("SELECT * FROM accounts ORDER BY id DESC")->fetchAll();

// Default form values (kept after submit or test failure)
$form = [
    'provider' => $_POST['provider'] ?? 'custom',
    'host' => $_POST['host'] ?? '',
    'port' => $_POST['port'] ?? 993,
    'encryption' => $_POST['encryption'] ?? 'ssl',
    'username' => $_POST['username'] ?? '',
    'forward_all' => isset($_POST['forward_all']),
    'keywords' => $_POST['keywords'] ?? ''
];
?>

<h1 class="mt-4"><i class="bi bi-mailbox2"></i> Email Accounts</h1>
<ol class="breadcrumb mb-4">
    <li class="breadcrumb-item"><a href="index.php">Dashboard</a></li>
    <li class="breadcrumb-item active">Accounts</li>
</ol>

<?= $message ?>

<div class="card mb-4">
    <div class="card-header d-flex justify-content-between">
        <div><i class="bi bi-plus-circle"></i> Add New Account</div>
        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#accountModal">
            <i class="bi bi-plus-lg"></i> Add Account
        </button>
    </div>
</div>

<!-- Accounts Table -->
<div class="card mb-4">
    <div class="card-header">
        <i class="bi bi-list-ul"></i> Managed Accounts
    </div>
    <div class="card-body">
        <div class="table-responsive">
            <table class="table table-hover" id="accountsTable">
                <thead class="table-light">
                    <tr>
                        <th>ID</th>
                        <th>Provider</th>
                        <th>Username</th>
                        <th>Host</th>
                        <th>Status</th>
                        <th>Rules</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($accounts as $acc):
                        $rules = json_decode($acc['rules'] ?? '{}', true);
                        $rule_text = $rules['forward_all'] ?? false ? 'Forward All' : 
                                   (isset($rules['keywords']) ? count($rules['keywords']) . ' keywords' : 'None');
                    ?>
                        <tr>
                            <td><?= $acc['id'] ?></td>
                            <td><?= htmlspecialchars($acc['provider'] ?? 'Custom') ?></td>
                            <td><?= htmlspecialchars($acc['username']) ?></td>
                            <td><?= htmlspecialchars($acc['host'] . ':' . $acc['port']) ?></td>
                            <td>
                                <form method="post" class="d-inline">
                                    <input type="hidden" name="action" value="toggle">
                                    <input type="hidden" name="id" value="<?= $acc['id'] ?>">
                                    <input type="hidden" name="active" value="<?= $acc['active'] ?>">
                                    <button type="submit" class="btn btn-sm <?= $acc['active'] ? 'btn-success' : 'btn-secondary' ?>">
                                        <?= $acc['active'] ? 'Active' : 'Inactive' ?>
                                    </button>
                                </form>
                            </td>
                            <td><?= $rule_text ?></td>
                            <td>
                                <button class="btn btn-sm btn-warning edit-btn"
                                        data-id="<?= $acc['id'] ?>"
                                        data-provider="<?= htmlspecialchars($acc['provider'] ?? '') ?>"
                                        data-host="<?= htmlspecialchars($acc['host']) ?>"
                                        data-port="<?= $acc['port'] ?>"
                                        data-encryption="<?= $acc['encryption'] ?>"
                                        data-username="<?= htmlspecialchars($acc['username']) ?>"
                                        data-rules='<?= htmlspecialchars($acc['rules'] ?? '{}') ?>'>
                                    Edit
                                </button>
                                <form method="post" class="d-inline">
                                    <input type="hidden" name="action" value="delete">
                                    <input type="hidden" name="id" value="<?= $acc['id'] ?>">
                                    <button type="submit" class="btn btn-sm btn-danger" onclick="return confirm('Delete this account?')">
                                        Delete
                                    </button>
                                </form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Add/Edit Modal -->
<div class="modal fade" id="accountModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <form method="post" id="accountForm">
                <div class="modal-header">
                    <h5 class="modal-title"><i class="bi bi-mailbox2"></i> <span id="modalTitle">Add Account</span></h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <input type="hidden" name="action" value="add" id="formAction">
                    <input type="hidden" name="id" id="accountId">

                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Provider Preset</label>
                            <select class="form-select" id="providerPreset">
                                <option value="custom">Custom / Manual</option>
                                <?php foreach ($presets as $key => $val): ?>
                                    <option value="<?= $key ?>" <?= $form['provider'] === $key ? 'selected' : '' ?>>
                                        <?= ucwords(str_replace('_', ' ', $key)) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Username/Email</label>
                            <input type="text" name="username" id="username" class="form-control" value="<?= htmlspecialchars($form['username']) ?>" required>
                        </div>
                    </div>

                    <div class="row g-3 mt-2">
                        <div class="col-md-5">
                            <label class="form-label">IMAP Host</label>
                            <input type="text" name="host" id="host" class="form-control" value="<?= htmlspecialchars($form['host']) ?>" required>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">Port</label>
                            <input type="number" name="port" id="port" class="form-control" value="<?= $form['port'] ?>" required>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Encryption</label>
                            <select name="encryption" id="encryption" class="form-select">
                                <option value="ssl" <?= $form['encryption'] === 'ssl' ? 'selected' : '' ?>>SSL</option>
                                <option value="tls" <?= $form['encryption'] === 'tls' ? 'selected' : '' ?>>TLS</option>
                                <option value="none" <?= $form['encryption'] === 'none' ? 'selected' : '' ?>>None</option>
                            </select>
                        </div>
                    </div>

                    <div class="mt-3">
                        <label class="form-label">Password / App Password</label>
                        <input type="password" name="password" id="password" class="form-control" value="<?= htmlspecialchars($form['password'] ?? '') ?>" required>
                    </div>

                    <div class="mt-4">
                        <label class="form-label">Forwarding Rules</label>
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="forward_all" name="forward_all" <?= $form['forward_all'] ? 'checked' : '' ?>>
                            <label class="form-check-label">Forward All Emails</label>
                        </div>
                        <div class="mt-2" id="keywordsGroup" style="display: <?= $form['forward_all'] ? 'none' : 'block' ?>;">
                            <label class="form-label">Or Forward Only If Subject/Body Contains (one per line)</label>
                            <textarea name="keywords" id="keywords" class="form-control" rows="4"><?= htmlspecialchars($form['keywords']) ?></textarea>
                        </div>
                    </div>

                    <div class="mt-4 text-center">
                        <button type="button" class="btn btn-info btn-lg" id="testConnectionBtn">
                            <i class="bi bi-wifi"></i> Test Connection
                        </button>
                        <div id="testResult" class="mt-3"></div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-success btn-lg">Save Account</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function () {
    const modal = document.getElementById('accountModal');
    const form = document.getElementById('accountForm');
    const testBtn = document.getElementById('testConnectionBtn');
    const resultEl = document.getElementById('testResult');

    new DataTable('#accountsTable');

    // Preset auto-fill
    document.getElementById('providerPreset').addEventListener('change', function () {
        const presets = <?= json_encode($presets) ?>;
        if (presets[this.value]) {
            document.getElementById('host').value = presets[this.value].host;
            document.getElementById('port').value = presets[this.value].port;
            document.getElementById('encryption').value = presets[this.value].encryption;
        }
    });

    // Rules toggle
    document.getElementById('forward_all').addEventListener('change', function () {
        document.getElementById('keywordsGroup').style.display = this.checked ? 'none' : 'block';
    });

    // Edit button
    document.querySelectorAll('.edit-btn').forEach(btn => {
        btn.addEventListener('click', function () {
            document.getElementById('modalTitle').textContent = 'Edit Account';
            document.getElementById('formAction').value = 'edit';
            document.getElementById('accountId').value = this.dataset.id;
            document.getElementById('providerPreset').value = this.dataset.provider || 'custom';
            document.getElementById('host').value = this.dataset.host;
            document.getElementById('port').value = this.dataset.port;
            document.getElementById('encryption').value = this.dataset.encryption;
            document.getElementById('username').value = this.dataset.username;

            try {
                const rules = JSON.parse(this.dataset.rules || '{}');
                document.getElementById('forward_all').checked = !!rules.forward_all;
                document.getElementById('keywordsGroup').style.display = rules.forward_all ? 'none' : 'block';
                document.getElementById('keywords').value = Array.isArray(rules.keywords) ? rules.keywords.join('\n') : '';
            } catch (e) {
                document.getElementById('forward_all').checked = true;
                document.getElementById('keywordsGroup').style.display = 'none';
            }

            resultEl.innerHTML = '';
            new bootstrap.Modal(modal).show();
        });
    });

    // Test Connection
    testBtn.addEventListener('click', function () {
        const formData = new FormData(form);
        resultEl.innerHTML = '<div class="spinner-border text-primary" role="status"><span class="visually-hidden">Testing...</span></div><span class="ms-2">Testing connection...</span>';

        fetch('<?= BASE_URL ?>api/test_account.php', {
            method: 'POST',
            body: formData
        })
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                resultEl.innerHTML = '<div class="alert alert-success"><i class="bi bi-check-circle"></i> ' + data.message + '</div>';
            } else {
                resultEl.innerHTML = '<div class="alert alert-danger"><i class="bi bi-x-circle"></i> ' + data.message + 
                    (data.debug ? '<br><small>' + data.debug + '</small>' : '') + '</div>';
            }
        })
        .catch(err => {
            resultEl.innerHTML = '<div class="alert alert-danger">Network error. Check console.</div>';
            console.error(err);
        });
    });

    // Prevent form reset on modal close
    modal.addEventListener('hidden.bs.modal', function () {
        // Keep values for retry
    });
});
</script>

<?php require_once 'includes/footer.php'; ?>