<?php
// forwarder.php - Auto-only mode, Message-ID tracking, log only forwarded emails, limit log to 50 entries
// FULLY FIXED: SMTP uses sender_name/sender_email | Fallback uses real domain (delivers to Gmail!)
require_once __DIR__ . '/config.php';
require_once __DIR__ . '/includes/db.php';
require_once __DIR__ . '/includes/functions.php';
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;
require_once __DIR__ . '/includes/PHPMailer/PHPMailer.php';
require_once __DIR__ . '/includes/PHPMailer/SMTP.php';
require_once __DIR__ . '/includes/PHPMailer/Exception.php';
set_time_limit(300);
ini_set('memory_limit', '512M');

// FIXED: Use your real domain from BASE_URL → delivers to Gmail!
$server_domain = parse_url(BASE_URL, PHP_URL_HOST) ?: 'yourdomain.com';
$server_email = get_setting('default_from_email', 'support@' . $server_domain);
$server_name = get_setting('default_from_name', 'Support');

// Fetch active accounts
$accounts = $pdo->query("SELECT * FROM accounts WHERE active = 1")->fetchAll();
if (!$accounts) exit;

// Fetch forwarding emails
$forwards = $pdo->query("SELECT email FROM forwards")->fetchAll(PDO::FETCH_COLUMN);
if (!$forwards) exit;

// Fetch active SMTPs
$smtps = $pdo->query("SELECT * FROM smtps WHERE active = 1 ORDER BY id ASC")->fetchAll();
$rotation_type = get_setting('rotation_type', 'round-robin');
$rotation_index = (int)get_setting('smtp_rotation_index', 0);

// Ensure table exists for tracking forwarded Message-IDs
$pdo->exec("
CREATE TABLE IF NOT EXISTS forwarded_messages (
    id INT AUTO_INCREMENT PRIMARY KEY,
    account_id INT NOT NULL,
    message_id VARCHAR(255) NOT NULL,
    forwarded_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(account_id, message_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
");

// Track if any email was forwarded in this run
$any_forwarded = false;

// Loop accounts
foreach ($accounts as $account) {
    $password = decrypt_data($account['password_encrypted']);
    if (!$password) continue;

    $account_email = $account['username'];
    $mailbox = '{' . $account['host'] . ':' . $account['port'] . '/imap/' . $account['encryption'] . '/novalidate-cert}INBOX';
    $connection = @imap_open($mailbox, $account['username'], $password, 0, 1);
    if (!$connection) continue;

    // Fetch all UIDs
    $uids = imap_search($connection, 'ALL', SE_UID);
    if (!$uids) {
        imap_close($connection);
        continue;
    }

    rsort($uids); // newest first
    // Only newest 5 for new accounts, otherwise only new messages
    $last_uid = (int)($account['last_uid'] ?? 0);
    if ($last_uid === 0) {
        $uids = array_slice($uids, 0, 5); // first load: newest 5
    } else {
        $uids = array_filter($uids, fn($uid) => $uid > $last_uid); // only new messages
    }

    foreach ($uids as $uid) {
        $overview = imap_fetch_overview($connection, $uid, FT_UID);
        if (!$overview) continue;

        $subject = imap_utf8($overview[0]->subject ?? 'No Subject');
        $from = imap_utf8($overview[0]->from ?? 'Unknown');
        $message_id = trim($overview[0]->message_id ?? '');

        // Skip if already forwarded
        if ($message_id) {
            $stmt = $pdo->prepare("SELECT 1 FROM forwarded_messages WHERE account_id = ? AND message_id = ?");
            $stmt->execute([$account['id'], $message_id]);
            if ($stmt->fetch()) continue;
        }

        // Parse sender
        $from_email = 'Unknown';
        $from_name = '';
        if (preg_match('/(.*)<(.+?)>/', $from, $m)) {
            $from_name = trim($m[1], '" ');
            $from_email = $m[2];
        } else {
            $from_email = trim($from, '<> ');
        }
        $sender_display = $from_name ? "$from_name <$from_email>" : $from_email;

        // Fetch message body and attachments
        $structure = imap_fetchstructure($connection, $uid, FT_UID);
        $parts = $structure->parts ?? [];
        $body = '';
        $attachments = [];
        if (!$parts) {
            $body = imap_body($connection, $uid, FT_UID | FT_PEEK);
        } else {
            foreach ($parts as $i => $part) {
                $part_number = $i + 1;
                $data = imap_fetchbody($connection, $uid, $part_number, FT_UID | FT_PEEK);
                if ($part->encoding == 3) $data = base64_decode($data);
                elseif ($part->encoding == 4) $data = quoted_printable_decode($data);
                $mime_type = strtolower($part->subtype ?? '');
                if ($mime_type === 'html') $body = $data;
                elseif ($part->ifdparameters && $part->dparameters) {
                    foreach ($part->dparameters as $dp) {
                        if (strtolower($dp->attribute) === 'filename') {
                            $attachments[] = ['name' => $dp->value, 'data' => $data];
                        }
                    }
                } elseif (!$body) {
                    $body = $data;
                }
            }
        }

        // Apply forwarding rules
        $rules = json_decode($account['rules'] ?? '{}', true);
        $forward_all = !empty($rules['forward_all']);
        $keywords = array_filter(array_map('trim', $rules['keywords'] ?? []));
        $should_forward = $forward_all;
        if (!$forward_all && $keywords) {
            $check_text = strtolower($body . ' ' . $subject);
            foreach ($keywords as $kw) {
                if (strpos($check_text, strtolower($kw)) !== false) {
                    $should_forward = true;
                    break;
                }
            }
        }
        if (!$should_forward) continue;

        $sent = false;
        $method = 'PHP mail()';
        // Format subject with account email
        $forward_subject = "FROM: $account_email - " . (stripos($subject, 'FWD:') === 0 ? $subject : "FWD: $subject");

        // Use SMTP if configured
        if ($smtps) {
            $selected = null;
            if ($rotation_type === 'random') $selected = $smtps[array_rand($smtps)];
            elseif ($rotation_type === 'round-robin') {
                $selected = $smtps[$rotation_index % count($smtps)];
                $rotation_index++;
                save_setting('smtp_rotation_index', $rotation_index);
            } elseif ($rotation_type === 'failover') {
                foreach ($smtps as $smtp) {
                    if (send_via_smtp($smtp, $forwards, $forward_subject, $body, $attachments)) {
                        $selected = $smtp;
                        $sent = true;
                        break;
                    }
                }
            }
            if ($selected && $rotation_type !== 'failover') {
                $sent = send_via_smtp($selected, $forwards, $forward_subject, $body, $attachments);
            }
            $method = $selected['host'] ?? $method;
        }

        // Fallback mail() — NOW USES REAL DOMAIN (delivers to Gmail!)
        if (!$sent) {
            $mail_headers = "From: $server_name <$server_email>\r\n";
            $mail_headers .= "Reply-To: $account_email\r\n";
            $mail_headers .= "Content-Type: text/html; charset=UTF-8";
            $sent = mail(implode(',', $forwards), $forward_subject, $body, $mail_headers);
        }

        if ($sent) {
            $any_forwarded = true;
            log_action(
                'info',
                "Forwarded to " . implode(', ', $forwards) . " via $method: \"$subject\" from $sender_display",
                $account['id'],
                $account_email,
                $subject
            );
            // Record forwarded message
            if ($message_id) {
                $stmt = $pdo->prepare("INSERT IGNORE INTO forwarded_messages (account_id, message_id) VALUES (?, ?)");
                $stmt->execute([$account['id'], $message_id]);
            }
            // Update last_uid immediately
            $pdo->prepare("UPDATE accounts SET last_uid = ? WHERE id = ?")->execute([$uid, $account['id']]);
        }
        unset($body, $attachments);
    }
    imap_close($connection);
}

// FIXED: Log cleanup using correct column
$pdo->exec("
DELETE FROM logs
WHERE id NOT IN (
    SELECT id FROM (
        SELECT id FROM logs ORDER BY timestamp DESC LIMIT 50
    ) AS keep
)
");

// NO completion log unless at least one email was forwarded
if ($any_forwarded) {
    log_action('debug', 'Forwarder script completed');
}

// FINAL: send_via_smtp uses your custom sender_name & sender_email from smtps table
function send_via_smtp($smtp, $to_emails, $subject, $body, $attachments = []) {
    $mail = new PHPMailer(true);
    try {
        $mail->isSMTP();
        $mail->Host = $smtp['host'];
        $mail->Port = $smtp['port'];
        $mail->SMTPAuth = !empty($smtp['username']);
        $mail->Username = $smtp['username'];
        $mail->Password = decrypt_data($smtp['password_encrypted']) ?: '';
        $mail->SMTPSecure = $smtp['encryption'] === 'none' ? false : $smtp['encryption'];

        // USE YOUR CUSTOM SENDER FROM DATABASE
        $from_email = !empty($smtp['sender_email']) ? $smtp['sender_email'] : ($smtp['username'] ?? 'no-reply@forwarder.local');
        $from_name  = !empty($smtp['sender_name']) ? $smtp['sender_name'] : 'Forwarder';

        $mail->setFrom($from_email, $from_name);
        $mail->addReplyTo($from_email, $from_name);

        foreach ($to_emails as $to) $mail->addAddress($to);
        $mail->isHTML(true);
        $mail->Subject = $subject;
        $mail->Body = $body;
        foreach ($attachments as $att) {
            $mail->addStringAttachment($att['data'], $att['name']);
        }
        return $mail->send();
    } catch (Exception $e) {
        log_action('error', "SMTP failed: " . $e->getMessage());
        return false;
    }
}