<?php
// includes/functions.php - Core Helper Functions
if (!defined('BASE_URL')) {
    die('Direct access not permitted');
}

// Secure session start (only if not already started)
function secure_session_start() {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
}

// Redirect with base URL
function redirect($path = '') {
    header("Location: " . BASE_URL . ltrim($path, '/'));
    exit;
}

// Check if user is logged in
function is_logged_in() {
    secure_session_start();
    return isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true;
}

// Force login
function require_login() {
    if (!is_logged_in()) {
        redirect('login.php');
    }
}

// Sanitize input
function sanitize($data) {
    return htmlspecialchars(trim($data), ENT_QUOTES, 'UTF-8');
}

// Encrypt data (passwords, tokens)
function encrypt_data($data) {
    if (empty($data)) return null;
    
    $iv = random_bytes(16); // 128-bit IV
    $encrypted = openssl_encrypt(
        $data,
        'aes-256-cbc',
        MASTER_KEY,
        0,
        $iv
    );
    
    if ($encrypted === false) {
        error_log("Encryption failed");
        return false;
    }
    
    // Return IV + encrypted data (base64 encoded)
    return base64_encode($iv . $encrypted);
}

// Decrypt data
function decrypt_data($data) {
    if (empty($data)) return null;
    
    $data = base64_decode($data);
    if ($data === false || strlen($data) < 16) return false;
    
    $iv = substr($data, 0, 16);
    $encrypted = substr($data, 16);
    
    $decrypted = openssl_decrypt(
        $encrypted,
        'aes-256-cbc',
        MASTER_KEY,
        0,
        $iv
    );
    
    return $decrypted !== false ? $decrypted : false;
}

// Custom Captcha (Math-based, no external API)
function generate_captcha() {
    secure_session_start();
    $a = rand(10, 30);
    $b = rand(5, 20);
    $_SESSION['captcha_answer'] = $a + $b;
    $_SESSION['captcha_time'] = time();
    return "$a + $b = ?";
}

function validate_captcha($input) {
    secure_session_start();
    if (!isset($_SESSION['captcha_answer']) || !isset($_SESSION['captcha_time'])) {
        return false;
    }
    // Expire after 5 minutes
    if (time() - $_SESSION['captcha_time'] > 300) {
        unset($_SESSION['captcha_answer'], $_SESSION['captcha_time']);
        return false;
    }
    $correct = $_SESSION['captcha_answer'];
    unset($_SESSION['captcha_answer'], $_SESSION['captcha_time']);
    return (int)$input === $correct;
}

// Log action (to database)
function log_action($level, $message, $account_id = null, $sender = null, $subject = null) {
    global $pdo;
    try {
        $stmt = $pdo->prepare("INSERT INTO logs (level, message, account_id, sender, subject) VALUES (?, ?, ?, ?, ?)");
        $stmt->execute([$level, $message, $account_id, $sender, $subject]);
        
        // Keep only last 500 logs (cleanup old ones)
        $pdo->query("DELETE FROM logs WHERE id NOT IN (SELECT id FROM (SELECT id FROM logs ORDER BY id DESC LIMIT 500) tmp)");
    } catch (Exception $e) {
        error_log("Log failed: " . $e->getMessage());
    }
}

// Get setting value
function get_setting($key, $default = null) {
    global $pdo;
    try {
        $stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
        $stmt->execute([$key]);
        $result = $stmt->fetchColumn();
        return $result !== false ? $result : $default;
    } catch (Exception $e) {
        return $default;
    }
}

// Save setting
function save_setting($key, $value) {
    global $pdo;
    try {
        $stmt = $pdo->prepare("INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = ?");
        $stmt->execute([$key, $value, $value]);
    } catch (Exception $e) {
        error_log("Save setting failed: " . $e->getMessage());
    }
}
?>