<?php
// index.php - Main Dashboard
require_once 'config.php';
require_once 'includes/db.php';
require_once 'includes/functions.php';
require_once 'includes/header.php';

// Fetch stats
$total_accounts = $pdo->query("SELECT COUNT(*) FROM accounts")->fetchColumn();
$active_accounts = $pdo->query("SELECT COUNT(*) FROM accounts WHERE active = 1")->fetchColumn();
$total_forwards = $pdo->query("SELECT COUNT(*) FROM forwards")->fetchColumn();
$recent_logs = $pdo->prepare("SELECT * FROM logs ORDER BY timestamp DESC LIMIT 10");
$recent_logs->execute();

// Fetch forward emails
$forward_emails = $pdo->query("SELECT email FROM forwards")->fetchAll(PDO::FETCH_COLUMN);
$forward_list = $forward_emails ? implode(', ', $forward_emails) : 'None configured';

// Get rotation type
$rotation_type = get_setting('rotation_type', 'round-robin');
?>

<h1 class="mt-4"><i class="bi bi-speedometer2"></i> Dashboard</h1>
<ol class="breadcrumb mb-4">
    <li class="breadcrumb-item active">Overview</li>
</ol>

<!-- Stats Cards -->
<div class="row">
    <div class="col-xl-3 col-md-6">
        <div class="card bg-primary text-white mb-4">
            <div class="card-body">
                <i class="bi bi-mailbox2 me-2"></i>
                Total Accounts
                <div class="float-end fs-3"><?= $total_accounts ?></div>
            </div>
            <div class="card-footer d-flex align-items-center justify-content-between">
                <a class="small text-white stretched-link" href="accounts.php">View Details</a>
                <i class="bi bi-arrow-right"></i>
            </div>
        </div>
    </div>

    <div class="col-xl-3 col-md-6">
        <div class="card bg-success text-white mb-4">
            <div class="card-body">
                <i class="bi bi-check-circle me-2"></i>
                Active Accounts
                <div class="float-end fs-3"><?= $active_accounts ?></div>
            </div>
            <div class="card-footer d-flex align-items-center justify-content-between">
                <a class="small text-white stretched-link" href="accounts.php">Manage</a>
                <i class="bi bi-arrow-right"></i>
            </div>
        </div>
    </div>

    <div class="col-xl-3 col-md-6">
        <div class="card bg-warning text-white mb-4">
            <div class="card-body">
                <i class="bi bi-forward me-2"></i>
                Forward To
                <div class="float-end fs-3"><?= $total_forwards ?></div>
            </div>
            <div class="card-footer d-flex align-items-center justify-content-between">
                <a class="small text-white stretched-link" href="forwards.php">Configure</a>
                <i class="bi bi-arrow-right"></i>
            </div>
        </div>
    </div>

    <div class="col-xl-3 col-md-6">
        <div class="card bg-info text-white mb-4">
            <div class="card-body">
                <i class="bi bi-send me-2"></i>
                SMTP Rotation
                <div class="float-end text-capitalize"><?= htmlspecialchars($rotation_type) ?></div>
            </div>
            <div class="card-footer d-flex align-items-center justify-content-between">
                <a class="small text-white stretched-link" href="smtps.php">Setup SMTPs</a>
                <i class="bi bi-arrow-right"></i>
            </div>
        </div>
    </div>
</div>

<!-- Account Status + Forward List -->
<div class="row">
    <div class="col-lg-6">
        <div class="card mb-4">
            <div class="card-header">
                <i class="bi bi-mailbox2"></i> Account Connection Status
                <span class="float-end small" id="statusLastUpdate">Checking...</span>
            </div>
            <div class="card-body" id="accountStatus">
                <p class="text-center text-muted"><i class="bi bi-hourglass-split"></i> Loading status...</p>
            </div>
        </div>
    </div>

    <div class="col-lg-6">
        <div class="card mb-4">
            <div class="card-header">
                <i class="bi bi-forward"></i> Forwarding Destinations
            </div>
            <div class="card-body">
                <?php if ($forward_list !== 'None configured'): ?>
                    <div class="alert alert-success">
                        <strong>Emails will be forwarded to:</strong><br>
                        <?= nl2br(htmlspecialchars($forward_list)) ?>
                    </div>
                <?php else: ?>
                    <div class="alert alert-warning">
                        <i class="bi bi-exclamation-triangle"></i>
                        No forwarding emails configured yet.
                    </div>
                <?php endif; ?>
                <a href="forwards.php" class="btn btn-primary">
                    <i class="bi bi-plus-circle"></i> Manage Forward Emails
                </a>
            </div>
        </div>
    </div>
</div>

<!-- Recent Logs -->
<div class="card mb-4">
    <div class="card-header">
        <i class="bi bi-journal-text"></i> Recent Forwarding Activity
    </div>
    <div class="card-body">
        <?php if ($recent_logs->rowCount() > 0): ?>
            <div class="table-responsive">
                <table class="table table-hover align-middle">
                    <thead>
                        <tr>
                            <th>Time</th>
                            <th>Level</th>
                            <th>Account</th>
                            <th>From</th>
                            <th>Subject</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php while ($log = $recent_logs->fetch()): ?>
                            <tr>
                                <td><?= date('M j, H:i', strtotime($log['timestamp'])) ?></td>
                                <td>
                                    <span class="badge bg-<?= $log['level'] === 'error' ? 'danger' : ($log['level'] === 'debug' ? 'secondary' : 'info') ?>">
                                        <?= strtoupper($log['level']) ?>
                                    </span>
                                </td>
                                <td><?= $log['account_id'] ? "Account #{$log['account_id']}" : '-' ?></td>
                                <td><?= htmlspecialchars($log['sender'] ?? 'N/A') ?></td>
                                <td><?= htmlspecialchars($log['subject'] ?? 'No subject') ?></td>
                            </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            </div>
            <a href="logs.php" class="btn btn-outline-secondary btn-sm">View All Logs →</a>
        <?php else: ?>
            <p class="text-muted text-center">No activity yet. Add accounts and start forwarding!</p>
        <?php endif; ?>
    </div>
</div>

<script>
// Real-time Account Status via AJAX
function updateAccountStatus() {
    fetch('<?= BASE_URL ?>api/status.php')
        .then(response => response.json())
        .then(data => {
            let html = '<ul class="list-group">';
            let allActive = true;
            data.forEach(acc => {
                const statusClass = acc.status === 'active' ? 'status-active' : 'status-inactive';
                const statusText = acc.status === 'active' ? 'Connected' : 'Disconnected';
                html += `
                    <li class="list-group-item d-flex justify-content-between align-items-center">
                        Account #${acc.id} (${acc.username})
                        <span>
                            <span class="status-dot ${statusClass}"></span>
                            ${statusText}
                        </span>
                    </li>`;
                if (acc.status !== 'active') allActive = false;
            });
            if (data.length === 0) {
                html += '<li class="list-group-item text-center text-muted">No accounts configured</li>';
            }
            html += '</ul>';
            document.getElementById('accountStatus').innerHTML = html;
            document.getElementById('statusLastUpdate').textContent = 'Updated: ' + new Date().toLocaleTimeString();
            
            // Optional: Show notification if any disconnected
            if (!allActive && data.length > 0) {
                document.getElementById('accountStatus').insertAdjacentHTML('beforeend', 
                    '<div class="alert alert-warning mt-3"><i class="bi bi-exclamation-triangle"></i> Some accounts are disconnected. Check <a href="accounts.php">Accounts</a>.</div>'
                );
            }
        })
        .catch(err => {
            document.getElementById('accountStatus').innerHTML = '<p class="text-danger">Failed to load status</p>';
        });
}

// Initial load + every 30 seconds
updateAccountStatus();
setInterval(updateAccountStatus, 30000);
</script>

<?php require_once 'includes/footer.php'; ?>