<?php
// login.php - Secure Login Page
require_once 'config.php';
require_once 'includes/db.php';
require_once 'includes/functions.php';

secure_session_start();

// If already logged in, redirect to dashboard
if (is_logged_in()) {
    redirect('index.php');
}

$error = '';
$username = '';

// Rate limiting: Max 5 attempts per 5 minutes
$max_attempts = 5;
$lockout_time = 300; // 5 minutes

if (isset($_SESSION['login_attempts']) && $_SESSION['login_attempts'] >= $max_attempts) {
    if (time() - $_SESSION['last_attempt_time'] < $lockout_time) {
        $error = "Too many failed attempts. Try again in " . ($lockout_time - (time() - $_SESSION['last_attempt_time'])) . " seconds.";
    } else {
        // Reset after lockout
        unset($_SESSION['login_attempts'], $_SESSION['last_attempt_time']);
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (empty($error)) {
        $username = sanitize($_POST['username'] ?? '');
        $password = $_POST['password'] ?? '';
        $captcha = $_POST['captcha'] ?? '';

        if (empty($username) || empty($password) || empty($captcha)) {
            $error = "All fields are required.";
        } elseif (!validate_captcha($captcha)) {
            $error = "Incorrect captcha answer.";
        } else {
            try {
                $stmt = $pdo->prepare("SELECT id, username, password_hash FROM users WHERE username = ? LIMIT 1");
                $stmt->execute([$username]);
                $user = $stmt->fetch();

                if ($user && password_verify($password, $user['password_hash'])) {
                    // Success
                    $_SESSION['logged_in'] = true;
                    $_SESSION['user_id'] = $user['id'];
                    $_SESSION['username'] = $user['username'];
                    unset($_SESSION['login_attempts'], $_SESSION['last_attempt_time']);
                    log_action('info', "Successful login: {$username}");
                    redirect('index.php');
                } else {
                    $error = "Invalid username or password.";
                    $_SESSION['login_attempts'] = ($_SESSION['login_attempts'] ?? 0) + 1;
                    $_SESSION['last_attempt_time'] = time();
                    log_action('error', "Failed login attempt for: {$username}");
                }
            } catch (Exception $e) {
                error_log("Login error: " . $e->getMessage());
                $error = "An error occurred. Try again.";
            }
        }
    }
}

// Generate new captcha
$captcha_question = generate_captcha();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - Email Forwarder Panel</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        :root {
            --bg-gradient: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            --card-bg: rgba(255, 255, 255, 0.95);
        }
        .dark-mode {
            --bg-gradient: linear-gradient(135deg, #232526 0%, #414345 100%);
            --card-bg: rgba(52, 58, 64, 0.95);
        }
        body {
            background: var(--bg-gradient);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: background 0.5s;
        }
        .login-card {
            max-width: 420px;
            border-radius: 15px;
            overflow: hidden;
            background: var(--card-bg);
        }
        .captcha-question {
            font-size: 1.3em;
            font-weight: bold;
            letter-spacing: 2px;
        }
        .theme-toggle {
            position: absolute;
            top: 20px;
            right: 20px;
            z-index: 10;
        }
    </style>
</head>
<body>

<button class="btn btn-outline-light theme-toggle" id="themeToggle">
    <i class="bi bi-moon-stars-fill" id="themeIcon"></i>
</button>

<div class="container">
    <div class="login-card card shadow-lg">
        <div class="card-header text-center bg-primary text-white py-4">
            <h3 class="mb-0"><i class="bi bi-shield-lock"></i> Admin Login</h3>
            <small>Email Forwarder Panel</small>
        </div>
        <div class="card-body p-5">
            <?php if ($error): ?>
                <div class="alert alert-danger">
                    <i class="bi bi-exclamation-triangle"></i> <?= htmlspecialchars($error) ?>
                </div>
            <?php endif; ?>

            <form method="post">
                <div class="mb-4">
                    <label class="form-label"><i class="bi bi-person"></i> Username</label>
                    <input type="text" name="username" class="form-control form-control-lg" value="<?= htmlspecialchars($username) ?>" required autofocus>
                </div>

                <div class="mb-4">
                    <label class="form-label"><i class="bi bi-lock"></i> Password</label>
                    <input type="password" name="password" class="form-control form-control-lg" required>
                </div>

                <div class="mb-4">
                    <label class="form-label"><i class="bi bi-shield-check"></i> Security Check</label>
                    <div class="captcha-question text-center mb-3 p-3 bg-light rounded">
                        <?= htmlspecialchars($captcha_question) ?>
                    </div>
                    <input type="text" name="captcha" class="form-control form-control-lg text-center" placeholder="Enter answer" required>
                </div>

                <div class="d-grid">
                    <button type="submit" class="btn btn-primary btn-lg">
                        <i class="bi bi-box-arrow-in-right"></i> Login
                    </button>
                </div>
            </form>
        </div>
        <div class="card-footer text-center text-muted small">
            © 2025 Email Forwarder Panel • Secure Access Only
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
    // Theme Toggle
    const toggle = document.getElementById('themeToggle');
    const icon = document.getElementById('themeIcon');
    const body = document.body;

    const savedTheme = localStorage.getItem('theme') || 'light';
    if (savedTheme === 'dark') {
        body.classList.add('dark-mode');
        icon.classList.replace('bi-moon-stars-fill', 'bi-sun-fill');
    }

    toggle.addEventListener('click', () => {
        body.classList.toggle('dark-mode');
        const isDark = body.classList.contains('dark-mode');
        icon.classList.toggle('bi-moon-stars-fill', !isDark);
        icon.classList.toggle('bi-sun-fill', isDark);
        localStorage.setItem('theme', isDark ? 'dark' : 'light');
    });
</script>
</body>
</html>