<?php
// logs.php - Improved Logs Table with Full Details
require_once 'config.php';
require_once 'includes/db.php';
require_once 'includes/functions.php';
require_once 'includes/header.php';

$message = '';

if (isset($_POST['clear_logs'])) {
    $pdo->exec("TRUNCATE TABLE logs");
    $message = '<div class="alert alert-success">Logs cleared!</div>';
}

$page = max(1, (int)($_GET['page'] ?? 1));
$per_page = 50;
$offset = ($page - 1) * $per_page;

$total = $pdo->query("SELECT COUNT(*) FROM logs")->fetchColumn();
$total_pages = ceil($total / $per_page);

$logs = $pdo->prepare("SELECT l.*, a.username as account_email FROM logs l LEFT JOIN accounts a ON l.account_id = a.id ORDER BY l.timestamp DESC LIMIT ? OFFSET ?");
$logs->execute([$per_page, $offset]);
?>

<h1 class="mt-4"><i class="bi bi-journal-text"></i> Forwarding Logs</h1>
<ol class="breadcrumb mb-4">
    <li class="breadcrumb-item"><a href="index.php">Dashboard</a></li>
    <li class="breadcrumb-item active">Logs</li>
</ol>

<?= $message ?>

<div class="card mb-4">
    <div class="card-header d-flex justify-content-between align-items-center">
        <span><i class="bi bi-clock-history"></i> Forwarding Activity (<?= $total ?> entries)</span>
        <?php if ($total > 0): ?>
            <form method="post" class="d-inline">
                <button type="submit" name="clear_logs" class="btn btn-outline-danger btn-sm" onclick="return confirm('Clear all logs?')">
                    <i class="bi bi-trash"></i> Clear Logs
                </button>
            </form>
        <?php endif; ?>
    </div>
    <div class="card-body">
        <?php if ($logs->rowCount() > 0): ?>
            <div class="table-responsive">
                <table class="table table-hover" id="logsTable">
                    <thead class="table-light">
                        <tr>
                            <th>Time</th>
                            <th>Account</th>
                            <th>From Name</th>
                            <th>From Email</th>
                            <th>Subject</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php while ($log = $logs->fetch()):
                            $is_success = strpos($log['message'], 'Forwarded') === 0 || strpos($log['message'], 'Forwarded to') === 0;
                            $status = $is_success ? 'Forwarded' : 'Failed';
                            $badge = $is_success ? 'success' : 'danger';
                        ?>
                            <tr>
                                <td><?= date('M d, H:i:s', strtotime($log['timestamp'])) ?></td>
                                <td><?= htmlspecialchars($log['account_email'] ?? 'Unknown') ?></td>
                                <td><?= htmlspecialchars($log['sender'] ?? 'Unknown') ?></td>
                                <td><?= htmlspecialchars($log['sender'] ?? 'Unknown') ?></td>
                                <td><?= htmlspecialchars($log['subject'] ?? 'No Subject') ?></td>
                                <td><span class="badge bg-<?= $badge ?>"><?= htmlspecialchars($log['message'] ?? 'No details') ?></span></td>
                            </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            </div>

            <!-- Pagination -->
            <?php if ($total_pages > 1): ?>
                <nav>
                    <ul class="pagination justify-content-center">
                        <li class="page-item <?= $page <= 1 ? 'disabled' : '' ?>">
                            <a class="page-link" href="?page=<?= $page - 1 ?>">Previous</a>
                        </li>
                        <?php for ($i = max(1, $page - 2); $i <= min($total_pages, $page + 2); $i++): ?>
                            <li class="page-item <?= $i === $page ? 'active' : '' ?>">
                                <a class="page-link" href="?page=<?= $i ?>"><?= $i ?></a>
                            </li>
                        <?php endfor; ?>
                        <li class="page-item <?= $page >= $total_pages ? 'disabled' : '' ?>">
                            <a class="page-link" href="?page=<?= $page + 1 ?>">Next</a>
                        </li>
                    </ul>
                </nav>
            <?php endif; ?>
        <?php else: ?>
            <p class="text-center text-muted">No logs yet. Send a test email and run the forwarder.</p>
        <?php endif; ?>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function () {
    new DataTable('#logsTable', {
        order: [[0, 'desc']],
        pageLength: 25,
        searching: true,
        columnDefs: [
            { targets: 0, width: "140px" },
            { targets: 5, width: "100px", className: "text-center" }
        ]
    });
});
</script>

<?php require_once 'includes/footer.php'; ?>