<?php
// setup.php - Installation Wizard (Improved with Accurate Login URL)
ob_start();
session_start();

// Prevent access if already installed
if (file_exists('config.php') || file_exists('install_lock.txt')) {
    die('<h2>Installation already completed.</h2><p>Delete <code>setup.php</code> for security or access via <a href="login.php">login.php</a>.</p>');
}

$step = $_POST['step'] ?? 1;
$error = '';
$success = false;
$login_url = '';  // Will hold the correct login URL

if ($step == 2) {
    // Sanitize and validate input
    $db_host = trim($_POST['db_host'] ?? '');
    $db_user = trim($_POST['db_user'] ?? '');
    $db_pass = $_POST['db_pass'] ?? '';
    $db_name = trim($_POST['db_name'] ?? '');
    $admin_user = trim($_POST['admin_user'] ?? '');
    $admin_pass = trim($_POST['admin_pass'] ?? '');
    $admin_pass2 = $_POST['admin_pass2'] ?? '';
    $master_key = trim($_POST['master_key'] ?? '');

    if (empty($db_host) || empty($db_user) || empty($db_name) || empty($admin_user) || empty($admin_pass)) {
        $error = "All fields are required.";
    } elseif ($admin_pass !== $admin_pass2) {
        $error = "Passwords do not match.";
    } elseif (strlen($admin_pass) < 8) {
        $error = "Password must be at least 8 characters.";
    } elseif (empty($master_key)) {
        $error = "Master encryption key is required.";
    } else {
        // Test DB connection
        try {
            $pdo = new PDO("mysql:host=$db_host;dbname=$db_name", $db_user, $db_pass, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION
            ]);
        } catch (Exception $e) {
            $error = "Database connection failed: " . htmlspecialchars($e->getMessage());
        }

        if (empty($error)) {
            // Generate strong key if too short
            if (strlen($master_key) < 32) {
                $master_key = bin2hex(random_bytes(32));
            }

            // === Dynamically Build Correct Base URL ===
            $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') || $_SERVER['SERVER_PORT'] == 443 ? "https://" : "http://";
            $host = $_SERVER['HTTP_HOST'];
            $script_path = dirname($_SERVER['SCRIPT_NAME']);
            // Remove trailing slash if not root
            $base_url = rtrim($protocol . $host . $script_path, '/\\') . '/';

            // Build login URL
            $login_url = $base_url . 'login.php';

            // Create config.php
            $config_content = "<?php\n";
            $config_content .= "define('DB_HOST', '$db_host');\n";
            $config_content .= "define('DB_USER', '$db_user');\n";
            $config_content .= "define('DB_PASS', '$db_pass');\n";
            $config_content .= "define('DB_NAME', '$db_name');\n";
            $config_content .= "define('MASTER_KEY', '$master_key');\n";
            $config_content .= "define('BASE_URL', '$base_url');\n";  // Save for later use
            $config_content .= "?>\n";

            if (file_put_contents('config.php', $config_content) === false) {
                $error = "Failed to write config.php. Check folder permissions (755 or 775).";
            } else {
                // Proceed with table creation
                require 'config.php';
                require 'includes/db.php'; // Will be created next

                $queries = [
                    "CREATE TABLE IF NOT EXISTS users (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        username VARCHAR(50) UNIQUE NOT NULL,
                        password_hash VARCHAR(255) NOT NULL
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",

                    "CREATE TABLE IF NOT EXISTS accounts (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        type ENUM('imap','exchange','oauth') DEFAULT 'imap',
                        provider VARCHAR(50),
                        host VARCHAR(100) NOT NULL,
                        port INT DEFAULT 993,
                        encryption ENUM('ssl','tls','none') DEFAULT 'ssl',
                        username VARCHAR(100) NOT NULL,
                        password_encrypted BLOB,
                        oauth_token_encrypted BLOB,
                        active TINYINT(1) DEFAULT 1,
                        rules JSON,
                        last_checked DATETIME,
                        INDEX(active)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",

                    "CREATE TABLE IF NOT EXISTS smtps (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        host VARCHAR(100) NOT NULL,
                        port INT DEFAULT 587,
                        encryption ENUM('ssl','tls','none') DEFAULT 'tls',
                        username VARCHAR(100),
                        password_encrypted BLOB,
                        active TINYINT(1) DEFAULT 1
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",

                    "CREATE TABLE IF NOT EXISTS forwards (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        email VARCHAR(100) UNIQUE NOT NULL
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",

                    "CREATE TABLE IF NOT EXISTS logs (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        timestamp DATETIME DEFAULT CURRENT_TIMESTAMP,
                        level ENUM('debug','info','error') DEFAULT 'info',
                        account_id INT,
                        sender VARCHAR(100),
                        subject VARCHAR(255),
                        message TEXT
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",

                    "CREATE TABLE IF NOT EXISTS queue (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        account_id INT,
                        message_raw LONGTEXT,
                        status ENUM('pending','processing','done','failed') DEFAULT 'pending',
                        added DATETIME DEFAULT CURRENT_TIMESTAMP
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",

                    "CREATE TABLE IF NOT EXISTS settings (
                        setting_key VARCHAR(50) PRIMARY KEY,
                        setting_value TEXT
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4"
                ];

                try {
                    foreach ($queries as $q) {
                        $pdo->exec($q);
                    }

                    // Insert admin user
                    $hash = password_hash($admin_pass, PASSWORD_DEFAULT);
                    $stmt = $pdo->prepare("INSERT INTO users (username, password_hash) VALUES (?, ?)");
                    $stmt->execute([$admin_user, $hash]);

                    // Default settings
                    $pdo->prepare("INSERT IGNORE INTO settings (setting_key, setting_value) VALUES ('rotation_type', ?)")->execute(['round-robin']);
                    $pdo->prepare("INSERT IGNORE INTO settings (setting_key, setting_value) VALUES ('log_level', ?)")->execute(['info']);

                    // Create install lock
                    file_put_contents('install_lock.txt', 'Installed on ' . date('Y-m-d H:i:s'));

                    $success = true;
                } catch (Exception $e) {
                    $error = "Database setup failed: " . htmlspecialchars($e->getMessage());
                    @unlink('config.php'); // Cleanup on failure
                }
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Install - Email Forwarder Panel</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        body { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); min-height: 100vh; display: flex; align-items: center; }
        .setup-card { max-width: 600px; border-radius: 15px; overflow: hidden; }
        .master-key { font-family: 'Courier New', monospace; background: #f8f9fa; padding: 12px; border-radius: 8px; font-size: 1.1em; }
        .copy-btn { cursor: pointer; }
    </style>
</head>
<body>
<div class="container">
    <div class="setup-card card shadow-lg">
        <div class="card-header text-center bg-gradient text-white py-4" style="background: linear-gradient(135deg, #667eea, #764ba2);">
            <h3 class="mb-0"><i class="bi bi-shield-lock-fill"></i> Email Forwarder Panel - Secure Setup</h3>
        </div>
        <div class="card-body p-5">
            <?php if ($success): ?>
                <div class="text-center">
                    <i class="bi bi-check-circle-fill text-success display-1 mb-4"></i>
                    <h3 class="text-success">Installation Successful!</h3>
                    <div class="alert alert-info mt-4">
                        <strong>Admin Username:</strong> <?= htmlspecialchars($admin_user) ?><br>
                        <strong>Login URL:</strong> <a href="<?= htmlspecialchars($login_url) ?>" class="alert-link"><?= htmlspecialchars($login_url) ?></a>
                    </div>

                    <hr class="my-5">

                    <h5><i class="bi bi-clock-history"></i> Setup Cron Jobs (Required for 24/7 Operation)</h5>
                    <p class="text-muted">Add these lines in crontab or cPanel Cron Jobs:</p>
                    <div class="bg-dark text-white p-3 rounded mb-3 text-start font-monospace small">
                        * * * * * cd <?= htmlspecialchars(dirname(__DIR__)) ?> && php forwarder.php >/dev/null 2>&1<br>
                        0 */6 * * * cd <?= htmlspecialchars(dirname(__DIR__)) ?> && php health_checker.php >/dev/null 2>&1
                    </div>

                    <div class="alert alert-warning mt-4">
                        <strong>Security:</strong> Delete <code>setup.php</code> immediately after use!
                    </div>

                    <a href="<?= htmlspecialchars($login_url) ?>" class="btn btn-success btn-lg px-5 mt-4">
                        <i class="bi bi-box-arrow-in-right"></i> Go to Login
                    </a>
                </div>
            <?php else: ?>
                <?php if ($error): ?>
                    <div class="alert alert-danger"><i class="bi bi-exclamation-triangle"></i> <?= $error ?></div>
                <?php endif; ?>

                <form method="post">
                    <input type="hidden" name="step" value="2">
                    <h5 class="mb-4"><i class="bi bi-database"></i> Database Settings</h5>
                    <div class="row g-3">
                        <div class="col-12"><input type="text" name="db_host" class="form-control" placeholder="Host (usually localhost)" value="<?= htmlspecialchars($db_host ?? 'localhost') ?>" required></div>
                        <div class="col-12"><input type="text" name="db_user" class="form-control" placeholder="Username" required></div>
                        <div class="col-12"><input type="password" name="db_pass" class="form-control" placeholder="Password (can be empty)"></div>
                        <div class="col-12"><input type="text" name="db_name" class="form-control" placeholder="Database Name" required></div>
                    </div>

                    <hr class="my-5">

                    <h5 class="mb-4"><i class="bi bi-person-lock"></i> Admin Account</h5>
                    <div class="row g-3">
                        <div class="col-12"><input type="text" name="admin_user" class="form-control" placeholder="Admin Username" value="<?= htmlspecialchars($admin_user ?? '') ?>" required></div>
                        <div class="col-12"><input type="password" name="admin_pass" class="form-control" placeholder="Password (min 8 characters)" required></div>
                        <div class="col-12"><input type="password" name="admin_pass2" class="form-control" placeholder="Confirm Password" required></div>
                    </div>

                    <hr class="my-5">

                    <h5 class="mb-4"><i class="bi bi-key-fill"></i> Master Encryption Key</h5>
                    <div class="input-group mb-3">
                        <input type="text" name="master_key" class="form-control master-key" value="<?= htmlspecialchars($master_key ?: bin2hex(random_bytes(32))) ?>" readonly>
                        <button class="btn btn-outline-secondary copy-btn" type="button" onclick="navigator.clipboard.writeText(this.previousElementSibling.value)"><i class="bi bi-copy"></i></button>
                    </div>
                    <small class="text-danger"><strong>Save this key offline!</strong> It's used to encrypt all passwords. Cannot be recovered.</small>

                    <div class="text-center mt-5">
                        <button type="submit" class="btn btn-primary btn-lg px-5">
                            <i class="bi bi-download"></i> Install Now
                        </button>
                    </div>
                </form>
            <?php endif; ?>
        </div>
        <div class="card-footer text-center text-muted small">
            Email Forwarder Panel © 2025 • Secure & Lightweight
        </div>
    </div>
</div>

<script>
    // Auto-select key on click
    document.querySelectorAll('.master-key').forEach(el => {
        el.addEventListener('click', () => el.select());
    });
</script>
</body>
</html>
<?php ob_end_flush(); ?>