<?php
// test.php - Server Compatibility Checker for Email Forwarding Panel
// Features: Responsive GUI with Bootstrap 5, Light/Dark theme toggle, Test button, Results table

// Start output buffering to prevent headers already sent issues
ob_start();
session_start();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Server Compatibility Test - Email Forwarder Panel</title>
    <!-- Bootstrap 5 CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Bootstrap Icons -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        :root {
            --bg-color: #f8f9fa;
            --text-color: #212529;
            --card-bg: #ffffff;
        }
        .dark-mode {
            --bg-color: #212529;
            --text-color: #f8f9fa;
            --card-bg: #343a40;
        }
        body {
            background-color: var(--bg-color);
            color: var(--text-color);
            transition: background-color 0.3s, color 0.3s;
        }
        .card {
            background-color: var(--card-bg);
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        }
        .theme-toggle {
            position: fixed;
            top: 15px;
            right: 15px;
            z-index: 1000;
        }
        .result-pass { color: #28a745; }
        .result-fail { color: #dc3545; }
    </style>
</head>
<body class="pb-5">

<div class="container mt-5 pt-5">
    <div class="text-center mb-4">
        <h1 class="display-5 fw-bold">Email Forwarder Panel - Server Test</h1>
        <p class="lead">Check if your server meets all requirements before installation</p>
    </div>

    <div class="card mx-auto" style="max-width: 900px;">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0"><i class="bi bi-server"></i> Server Compatibility Checker</h5>
            <button id="themeToggle" class="btn btn-sm btn-outline-secondary theme-toggle">
                <i class="bi bi-moon-stars-fill" id="themeIcon"></i>
            </button>
        </div>
        <div class="card-body">
            <?php if (!isset($_POST['run_test'])): ?>
                <div class="text-center py-5">
                    <p class="text-muted">Click the button below to start testing your server environment.</p>
                    <form method="post">
                        <button type="submit" name="run_test" class="btn btn-primary btn-lg px-5">
                            <i class="bi bi-play-circle"></i> Run Tests
                        </button>
                    </form>
                </div>
            <?php else: ?>
                <h5 class="mb-4">Test Results</h5>
                <div class="table-responsive">
                    <table class="table table-bordered align-middle">
                        <thead class="table-light">
                            <tr>
                                <th width="40%">Requirement</th>
                                <th width="15%">Status</th>
                                <th width="45%">Details</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            $tests = [];

                            // 1. PHP Version (>= 7.4)
                            $php_ok = version_compare(PHP_VERSION, '7.4.0', '>=');
                            $tests[] = [
                                'name' => 'PHP Version ≥ 7.4',
                                'pass' => $php_ok,
                                'details' => 'Current: ' . PHP_VERSION . ($php_ok ? '' : ' → Upgrade required')
                            ];

                            // 2. Required Extensions
                            $exts = ['imap', 'openssl', 'pdo', 'pdo_mysql', 'curl', 'mbstring', 'json', 'session'];
                            foreach ($exts as $ext) {
                                $loaded = extension_loaded($ext);
                                $tests[] = [
                                    'name' => "Extension: $ext",
                                    'pass' => $loaded,
                                    'details' => $loaded ? 'Loaded' : 'Not loaded → Contact hosting or enable in php.ini'
                                ];
                            }

                            // 3. IMAP Extension (critical)
                            $imap_ok = extension_loaded('imap');
                            $tests[] = [
                                'name' => 'IMAP Extension (Critical)',
                                'pass' => $imap_ok,
                                'details' => $imap_ok ? 'Available' : 'Missing → Most shared hosts need to enable this (cPanel → Select PHP Version → Check imap)'
                            ];

                            // 4. OpenSSL (for encryption)
                            $openssl_ok = extension_loaded('openssl');
                            $tests[] = [
                                'name' => 'OpenSSL (Encryption)',
                                'pass' => $openssl_ok,
                                'details' => $openssl_ok ? 'Available' : 'Missing → Required for credential encryption'
                            ];

                            // 5. Write Permissions (current directory)
                            $write_ok = is_writable(__DIR__);
                            $tests[] = [
                                'name' => 'Write Permission (Current Folder)',
                                'pass' => $write_ok,
                                'details' => $write_ok ? 'Writable' : 'Not writable → CHMOD 755 or 775 needed for config files'
                            ];

                            // 6. Cron Job Support (detect cPanel or Linux cron)
                            $cron_ok = function_exists('exec') && !in_array('exec', array_map('trim', explode(',', ini_get('disable_functions'))));
                            $tests[] = [
                                'name' => 'Cron Jobs / exec() Function',
                                'pass' => $cron_ok,
                                'details' => $cron_ok
                                    ? 'exec() enabled → You can run forwarder.php via cron'
                                    : 'exec() disabled → Common on cheap shared hosts. Check with host or use alternative scheduling'
                            ];

                            // 7. Memory Limit
                            $memory = ini_get('memory_limit');
                            $memory_ok = (int)$memory >= 128 || $memory === '-1';
                            $tests[] = [
                                'name' => 'Memory Limit ≥ 128M',
                                'pass' => $memory_ok,
                                'details' => "Current: $memory" . ($memory_ok ? '' : ' → Increase in php.ini or .htaccess')
                            ];

                            // 8. Session Support
                            $session_ok = session_status() === PHP_SESSION_NONE || session_status() === PHP_SESSION_ACTIVE;
                            $tests[] = [
                                'name' => 'Session Support',
                                'pass' => true, // Almost always true
                                'details' => 'Working'
                            ];

                            // Output results
                            foreach ($tests as $test) {
                                $icon = $test['pass'] ? '<i class="bi bi-check-circle-fill result-pass"></i> PASS' : '<i class="bi bi-x-circle-fill result-fail"></i> FAIL';
                                $class = $test['pass'] ? 'table-success' : 'table-danger';
                                echo "<tr class=\"$class\">
                                    <td><strong>{$test['name']}</strong></td>
                                    <td class=\"text-center\">$icon</td>
                                    <td>{$test['details']}</td>
                                  </tr>";
                            }

                            // Final verdict
                            $all_pass = array_filter($tests, fn($t) => !$t['pass']) === [];
                            echo "<tr class=\"table-info fw-bold\">
                                <td colspan=\"3\" class=\"text-center\">
                                    " . ($all_pass
                                        ? '<span class="text-success"><i class="bi bi-trophy"></i> All tests passed! Your server is ready.</span>'
                                        : '<span class="text-danger"><i class="bi bi-exclamation-triangle"></i> Some issues detected. Fix them before installing.</span>') . "
                                </td>
                              </tr>";
                            ?>
                        </tbody>
                    </table>
                </div>

                <div class="mt-4 text-center">
                    <a href="test.php" class="btn btn-secondary">
                        <i class="bi bi-arrow-repeat"></i> Run Tests Again
                    </a>
                </div>
            <?php endif; ?>
        </div>
        <div class="card-footer text-muted small text-center">
            Email Forwarder Panel • Compatibility Checker • Built for PHP 7.4+
        </div>
    </div>
</div>

<!-- Bootstrap JS -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
    // Theme Toggle (Light/Dark Mode)
    const themeToggle = document.getElementById('themeToggle');
    const themeIcon = document.getElementById('themeIcon');
    const body = document.body;

    const currentTheme = localStorage.getItem('theme') || 'light';
    if (currentTheme === 'dark') {
        body.classList.add('dark-mode');
        themeIcon.classList.replace('bi-moon-stars-fill', 'bi-sun-fill');
    }

    themeToggle.addEventListener('click', () => {
        body.classList.toggle('dark-mode');
        const isDark = body.classList.contains('dark-mode');
        themeIcon.classList.toggle('bi-moon-stars-fill', !isDark);
        themeIcon.classList.toggle('bi-sun-fill', isDark);
        localStorage.setItem('theme', isDark ? 'dark' : 'light');
    });
</script>
</body>
</html>
<?php ob_end_flush(); ?>