<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Employee;
use App\leave;
use App\LeaveType;
use App\Notifications\EmployeeLeaveNotification; //Mail
use App\Notifications\LeaveNotification; //Database
use App\Notifications\LeaveNotificationToAdmin; //Database
use App\User;

use App\company;
use App\Holiday;
use App\department;
use App\office_shift;

use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Response;

class LeaveController extends Controller
{

    public function index(Request $request)
    {
        // kalau leaves.employee_id = employees.id:
        $employeeId = optional($request->user()->employee)->id ?? $request->user()->id;
    
        $perPage = (int) $request->input('limit', 10);
        $page    = (int) $request->input('page', 1);
    
        $leaves = \App\Leave::with(['employee:id,remaining_leave,leave_debt','LeaveType:id,leave_type'])
            ->where('employee_id', $employeeId)
            ->orderByDesc('created_at')
            ->paginate(
                $perPage,
                ['id','employee_id','leave_type_id','start_date','end_date','total_days','is_half','status','created_at','leave_reason'],
                'page',
                $page
            );
    
        return response()->json($leaves, 200);
    }



    public function list_approval(Request $request)
	{
		// Get the currently authenticated user
		$logged_user = auth()->user();
		$employee = Employee::find($logged_user->id);
        
        if($employee->supervisor == 0 && $employee->dept_head == 0)
        {
            $paginatedLeaves = array();
            return Response::json($paginatedLeaves, 200);
        }
        
		// Default pagination values
		$perPage = $request->input('limit', 10); // Default to 10 items per page
		$page = $request->input('page', 1); // Default to page 1

		// Start building the query with relationships
		$leave = Leave::with('employee', 'department', 'LeaveType');

		// Supervisor specific logic
		if (isset($employee->supervisor) && $employee->supervisor == 1) {
			$leave = $leave->whereHas('employee', function($query) {
				$query->where('dept_head', 0)->where('supervisor', 0)->where('coo', 0);
			})->where('department_id', $employee->department_id);
		}

		// Dept Head specific logic
		if (isset($employee->dept_head) && $employee->dept_head == 1) {
			$leave = $leave->whereHas('employee', function($query) {
				$query->where('dept_head', 0)->where('coo', 0);
			})->where('department_id', $employee->department_id);
		}

		// COO specific logic
		if (isset($employee->coo) && $employee->coo == 1) {
			$leave = $leave->whereHas('employee', function($query) {
				$query->where('coo', 0);
			})->where('company_id', $employee->company_id);
		}
    
		// Apply pagination
		$paginatedLeaves = $leave->paginate($perPage, ['*'], 'page', $page);

		// Return the paginated response as JSON
		return Response::json($paginatedLeaves, 200);
	}

    public function list_applications(Request $request)
    {
        $user = $request->user();
        // kalau id employee ≠ users.id, map sesuai skema kamu
        $employeeId = $user->id;
    
        $perPage = (int) $request->input('limit', 10);
        $page    = (int) $request->input('page', 1);
    
        $leaves = \App\Leave::with([
            // ALIAS ke camelCase agar cocok dengan model Flutter
            'employee' => function ($q) {
                $q->select(
                    'id',
                    \DB::raw('COALESCE(remaining_leave, 0) as remainingLeave'),
                    \DB::raw('COALESCE(leave_debt, 0) as leaveDebt')
                );
            },
            // opsional: title type
            'LeaveType:id,leave_type',
        ])
        ->where('employee_id', $employeeId)
        ->orderByDesc('created_at')
        ->paginate(
            $perPage,
            [
                'id',
                'employee_id',
                'leave_type_id',
                'start_date',
                'end_date',
                'total_days',
                'is_half',
                'status',
                'created_at',
                'leave_reason',
            ],
            'page',
            $page
        );
    
        return response()->json($leaves, 200);
    }


public function add(Request $request) {
        // [DEBUG] Start Logging
        Log::info('--- MULAI REQUEST PENGAJUAN CUTI ---');
        Log::info('User ID Login: ' . Auth::id());
        Log::info('Data Request dari Flutter:', $request->all());

        try {
            // 1. Validasi
            $validator = Validator::make(
                $request->only('leave_type', 'start_date', 'end_date', 'total_days'),
                [
                    'leave_type' => 'required',
                    'total_days' => 'required',
                    'start_date' => 'required',
                    'end_date'   => 'required|after_or_equal:start_date'
                ]
            );

            if ($validator->fails()) {
                Log::error('Validasi Gagal:', $validator->errors()->all());
                return response()->json(['success' => false, 'errors' => $validator->errors()->all()]);
            }

            // 2. Ambil Employee
            $employee = Employee::find(Auth::user()->id);
            
            if (!$employee) {
                Log::error('Employee tidak ditemukan untuk User ID: ' . Auth::id());
                return response()->json(['success' => false, 'errors' => ['Data Karyawan (Employee) tidak ditemukan.']]);
            }
            Log::info('Employee Ditemukan: ' . $employee->first_name . ' (ID: ' . $employee->id . ')');

            $leaveType = LeaveType::findOrFail($request->leave_type);
            Log::info('Tipe Cuti: ' . $leaveType->leave_type . ' (ID: ' . $leaveType->id . ')');

            // 3. Hitung Durasi (Logika yang diperbaiki)
            $diff_hidden = $request->input('diff_date_hidden');
            $req_total = $request->total_days;

            // Prioritaskan diff_date_hidden, jika null pakai total_days
            $current_leave_days = $diff_hidden ? (int)$diff_hidden : (int)$req_total;
            
            Log::info("Perhitungan Hari -> Hidden: $diff_hidden, TotalReq: $req_total, FINAL: $current_leave_days");

            // 4. Cek History Cuti (Untuk kuota tipe cuti)
            $history_leaves = leave::where('employee_id', $employee->id)
                                   ->where('leave_type_id', $request->leave_type)
                                   ->get();
            
            $previous_days_taken = $history_leaves->sum('total_days');
            $grand_total_usage = $previous_days_taken + $current_leave_days;

            Log::info("History Cuti: $previous_days_taken hari. Grand Total akan menjadi: $grand_total_usage hari.");

            // 5. Cek Allocated Quota (Batas Cuti Khusus)
            if ($leaveType->allocated_day != null) {
                Log::info("Cek Kuota Tipe Cuti: Batas " . $leaveType->allocated_day . " vs Terpakai " . $grand_total_usage);
                if ($leaveType->allocated_day < $grand_total_usage) {
                    $msg = 'Allocated quota exceeded.';
                    Log::warning($msg);
                    return response()->json(['success' => false, 'errors' => [$msg]]);
                }
            }

            // 6. Cek Sisa Cuti Tahunan (Remaining Leave)
            // Pastikan kita refresh data employee terbaru
            $employee_leave_info = Employee::find($employee->id);
            Log::info("Sisa Cuti Karyawan (Database): " . $employee_leave_info->remaining_leave);
            Log::info("Pengajuan Saat Ini: " . $current_leave_days);

            if ($current_leave_days > $employee_leave_info->remaining_leave) {
                Log::warning("Sisa cuti tidak cukup!");
                return response()->json(['success' => false, 'errors' => ["The employee's remaining leaves are insufficient"]]);
            }

            // 7. Simpan Data
            $data = [];
            $data['employee_id']   = $employee->id;
            $data['company_id']    = $employee->company_id;
            $data['department_id'] = $employee->department_id;
            $data['leave_type_id'] = $request->leave_type;
            $data['leave_reason']  = $request->leave_reason;
            $data['remarks']       = $request->remarks;
            $data['status']        = 'pending';
            $data['is_half']       = $request->is_half;
            $data['is_notify']     = $request->is_notify;
            $data['start_date']    = $request->start_date;
            $data['end_date']      = $request->end_date;
            $data['total_days']    = $current_leave_days;

            Log::info('Mencoba menyimpan data ke DB...', $data);

            $leave = leave::create($data);

            Log::info('Berhasil Disimpan! ID Leave: ' . $leave->id);

            // 8. Notifikasi (Kita bungkus try catch terpisah agar jika notif gagal, data tetap tersimpan)
            try {
                if ($leave->is_notify == 1) {
                    $text = "A new leave-notification has been requested by ". $employee_leave_info->first_name ." ".$employee_leave_info->last_name;
                    User::findOrFail($data['employee_id'])->notify(new LeaveNotification($text));
                    // ... (Kode notifikasi lain disederhanakan untuk debugging, biarkan logika asli Anda di sini jika mau)
                }
            } catch (\Exception $eNotify) {
                Log::error("Gagal Mengirim Notifikasi (Tapi data tersimpan): " . $eNotify->getMessage());
            }

            return response()->json(['success' => true, 'message' => 'Data Added successfully.']);

        } catch (\Exception $e) {
            // INI BAGIAN TERPENTING
            // Kita tangkap errornya, catat, dan kirim JSON (bukan HTML) ke Flutter
            Log::error("CRITICAL ERROR di function add: " . $e->getMessage());
            Log::error($e->getTraceAsString());

            return response()->json([
                'success' => false, 
                // Kirim pesan error asli ke Flutter agar muncul di Snack Bar
                'errors' => ['SERVER ERROR: ' . $e->getMessage()] 
            ], 500);
        }
    }
    
	public function update(Request $request)
	{
		$logged_user = auth()->user();

		if ($logged_user->can('edit-leave')) {
			$id = $request->id;
			$validator = Validator::make(
				$request->only(
                    'id',
					'employee_id',
					'status',
				),
				[   'id' => 'required',
					'employee_id' => 'required',
					'status' => 'required',
				]
			);
			if ($validator->fails()) {
				return response()->json(['errors' => $validator->errors()->all()]);
			}

			# Checking Supervior and manager
			if ($request->status) {
				$current_employee = Employee::find($logged_user->id);
				$requester_employee = Employee::find($request->employee_id);
				$checking_leave = leave::find($id);

				if(!$logged_user->hasRole('admin')) {
					if($current_employee->dept_head == 1 && $checking_leave->status != "first level approval" && $requester_employee->supervisor != 1)
					{
						return response()->json(['error' => __('Manager allowed to approved first level approval and supervisor request')]);
					}
					elseif($current_employee->supervisor == 1 && ($checking_leave->status != "pending" || $checking_leave->status == "first level approval"))
					{
					    return response()->json(['error' => __('Supervisor only allowed to first level approval')]);
					}
				}
				$data['status'] = $request->status;
			}

			$employee = leave::where('id', '!=', $id)
					->where('employee_id', $request->employee_id)->first();
			$leave = LeaveType::findOrFail($employee->leave_type_id);

			####CALCULATE TOTAL LEAVE DAYS###########
			$karyawan = Employee::find($request->employee_id);
			$start_date = Carbon::parse($employee->start_date);
			$end_date = Carbon::parse($employee->end_date);
			$office_shift = office_shift::find($karyawan->office_shift_id); // Asumsikan ada satu shift yang digunakan
			$workdays = [];
			if ($office_shift->monday_in) {
				$workdays[] = Carbon::MONDAY;
			}
			if ($office_shift->tuesday_in) {
				$workdays[] = Carbon::TUESDAY;
			}
			if ($office_shift->wednesday_in) {
				$workdays[] = Carbon::WEDNESDAY;
			}
			if ($office_shift->thursday_in) {
				$workdays[] = Carbon::THURSDAY;
			}
			if ($office_shift->friday_in) {
				$workdays[] = Carbon::FRIDAY;
			}
			if ($office_shift->saturday_in) {
				$workdays[] = Carbon::SATURDAY;
			}
			if ($office_shift->sunday_in) {
				$workdays[] = Carbon::SUNDAY;
			}

			// Ambil data hari libur nasional
			$holidays = Holiday::where(function ($query) use ($start_date, $end_date) {
				$query->whereBetween('start_date', [$start_date, $end_date])
					->orWhereBetween('end_date', [$start_date, $end_date])
					->orWhere(function ($query) use ($start_date, $end_date) {
						$query->where('start_date', '<=', $start_date)
							->where('end_date', '>=', $end_date);
					});
			})->get();

			$total_leave_days = 0;

			for ($date = $start_date; $date->lte($end_date); $date->addDay()) {
				if (in_array($date->dayOfWeek, $workdays)) {
					// Cek apakah tanggal ini merupakan hari libur nasional
					$isHoliday = $holidays->contains(function ($holiday) use ($date) {
						return $date->between($holiday->start_date, $holiday->end_date);
					});

					if (!$isHoliday) {
						$total_leave_days++;
					}
				}
			}
			########END CALCULATE TOTAL LEAVE##########
			
            $total_leave_days = $checking_leave->total_days; 
            
			if (!empty($employee)) {
				$total = 0;
				$employee_leaves = $employee->get();

				foreach ($employee_leaves as $employee_leave) {
					$total = $total + $employee_leave->total_days;
				}
				$total = $total + $total_leave_days;

				if ($leave->allocated_day != null && $leave->allocated_day < $total) {
					return response()->json(['limit' => __('Allocated quota for this leave type is less then requested days for this employee')]);
				}
			} else {
				if ($leave->allocated_day != null && $leave->allocated_day < $request->diff_date_hidden) {
					return response()->json(['limit' => __('Allocated quota for this leave type is less then requested days,You can select manual')]);
				}
			}

			$data = [];
			$data['employee_id'] = $request->employee_id;
			$data['status'] = $request->status ?? 'pending';

			//Employee Remaining Leave --- Start
			$employee_leave_info = Employee::find($request->employee_id);
			if ($total_leave_days > $employee_leave_info->remaining_leave && $employee->leave_type_id == 3 && $employee->leave_type_id == 5) {
				return response()->json(['remaining_leave' => "The employee's remaining leaves are insufficient"]);
			} elseif ($request->status == 'approved') {
				if($checking_leave->leave_type_id == 3 || $checking_leave->leave_type_id == 5)
				{
					$employee_leave_info->remaining_leave = $employee_leave_info->remaining_leave - $total_leave_days;
				}
				$employee_leave_info->update();
			}
			//Employee Remaining Leave  --- End

			leave::find($id)->update($data);

			$text = "Your notification has been ". $data['status'];
			$notifiable = User::findOrFail($data['employee_id']);
			$notifiable->notify(new LeaveNotification($text)); //To Employee
			return response()->json(['success' => __('Data is successfully updated')]);
		}
		return response()->json(['success' => __('You are not authorized')]);
	}

//     public function update(Request $request)
// 	{
// 		$logged_user = auth()->user();

// 		if ($logged_user->can('edit-leave')) {
// 			$id = $request->id;
// 			$validator = Validator::make(
// 				$request->only(
//                     'id',
// 					'employee_id',
// 					'status',
// 				),
// 				[   'id' => 'required',
// 					'employee_id' => 'required',
// 					'status' => 'required',
// 				]
// 			);
// 			if ($validator->fails()) {
// 				return response()->json(['errors' => $validator->errors()->all()]);
// 			}
// 			$data = [];
// 			$data['employee_id'] = $request->employee_id;
// 			$data['status'] = $request->status ?? 'pending';

// 			leave::find($id)->update($data);
// 			$text = "Your notification has been ". $data['status'];
// 			$notifiable = User::findOrFail($data['employee_id']);
// 			$notifiable->notify(new LeaveNotification($text)); //To Employee
// 			return response()->json(['success' => __('Data is successfully updated')]);
// 		}
// 		return response()->json(['success' => __('You are not authorized')]);
// 	}
    
    public function leave_type(Request $request) {
		$leave_types = LeaveType::all();
		return response([ 'data' => $leave_types, 'success' => true], 200);
	}

	public function approve(Request $request)
	{
		$request->validate([
			'id' => 'required|integer|exists:leaves,id',
		]);

		$leave = \App\leave::findOrFail($request->id);

		$previous = strtolower(trim((string) $leave->status));
		if ($previous === '') $previous = 'pending';

		// Role approver dari user login (7 = SPV, 4 = Manager)
		$user   = $request->user();
		$roleId = (int) ($user->role_users_id ?? 0);

		// ❗Jika sudah approved dan SPV klik approve lagi → gagal
		if ($roleId === 7 && $previous === 'approved') {
			return response()->json([
				'success' => false,
				'message' => 'Pengajuan sudah Approved dan tidak dapat di-approve ulang oleh SPV.',
			], 422);
		}

		// ❗Manager hanya boleh approve ketika statusnya FIRST LEVEL APPROVAL
		//    - kalau masih pending → tolak (harus SPV dulu)
		//    - kalau sudah approved → tolak (sudah selesai)
		if ($roleId === 4 && $previous === 'pending') {
			return response()->json([
				'success' => false,
				'message' => 'Status masih Pending. Harus First Level Approval oleh SPV terlebih dahulu.',
			], 422);
		}
		if ($roleId === 4 && $previous === 'approved') {
			return response()->json([
				'success' => false,
				'message' => 'Leave sudah Approved.',
			], 422);
		}

		// Tentukan transisi dari role
		$newStatus = null;
		if ($roleId === 7) { // SPV
			if ($previous === 'pending') {
				$newStatus = 'first level approval';
			} else {
				return response()->json([
					'success' => false,
					'message' => 'SPV hanya dapat menyetujui dari status Pending.',
				], 422);
			}
		} elseif ($roleId === 4) { // Manager
			if ($previous === 'first level approval') {
				$newStatus = 'approved';
			} else {
				return response()->json([
					'success' => false,
					'message' => 'Manager hanya dapat menyetujui dari First Level Approval.',
				], 422);
			}
		} else {
			return response()->json([
				'success' => false,
				'message' => 'Role tidak diizinkan melakukan approve.',
			], 403);
		}

		// Proses persetujuan (termasuk pengurangan jatah cuti saat Manager approve)
		return \DB::transaction(function () use ($leave, $newStatus, $previous) {
			// Jika MANAGER meng-approve ⇒ kurangi jatah cuti sesuai total_days untuk tipe tertentu (contoh: 3 & 5)
			if ($newStatus === 'approved') {
				$employee = \App\Employee::find($leave->employee_id);
				if ($employee) {
					$consumesQuota = in_array((int) $leave->leave_type_id, [3, 5], true);
					if ($consumesQuota) {
						$days = (int) $leave->total_days;

						if ($employee->remaining_leave < $days) {
							return response()->json([
								'success' => false,
								'message' => "The employee's remaining leaves are insufficient",
							], 422);
						}

						$employee->remaining_leave = $employee->remaining_leave - $days;
						$employee->save();
					}
				}
			}

			// Update status leave
			$leave->update(['status' => $newStatus]);

			// Notifikasi
			$text = "Your leave status has been {$newStatus}";
			if ($notifiable = \App\User::find($leave->employee_id)) {
				$notifiable->notify(new \App\Notifications\LeaveNotification($text));
			}

			return response()->json([
				'success' => true,
				'message' => 'Leave status updated',
				'data'    => [
					'id'              => $leave->id,
					'previous_status' => $previous,
					'new_status'      => $newStatus,
					'remaining_leave' => isset($employee) ? $employee->remaining_leave : null,
				],
			]);
		});
	}


	public function reject(Request $request)
	{
		$request->validate([
			'id'     => 'required|integer|exists:leaves,id',
			'status' => 'nullable|string|in:rejected', // default akan diisi 'rejected'
		]);

		$leave = leave::findOrFail($request->id);

		$previous = trim((string) $leave->status);
		$newStatus = $request->input('status', 'rejected'); // default rejected

		// HANYA ubah kolom status
		$leave->update(['status' => $newStatus]);

		// opsional: kirim notifikasi
		$text = "Your leave status has been {$newStatus}";
		$notifiable = User::find($leave->employee_id);
		if ($notifiable) {
			$notifiable->notify(new LeaveNotification($text));
		}

		return response()->json([
			'success' => true,
			'message' => 'Leave status updated',
			'data'    => [
				'id'              => $leave->id,
				'previous_status' => $previous,
				'new_status'      => $newStatus,
			],
		]);
	}

}
