<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

/**
 * Controller notifikasi berbasis channel "database".
 * Pastikan tabel `notifications` sudah ada (php artisan notifications:table && migrate).
 */
class NotificationController extends Controller
{
    /**
     * GET /api/notifications
     * Query:
     *  - page (default: 1)
     *  - limit (default: 10)
     *  - module (opsional, filter by data->module)
     *  - action (opsional, filter by data->action)
     */
    public function index(Request $request)
    {
        $perPage = (int) $request->input('limit', 10);
        $page    = (int) $request->input('page', 1);
        $module  = $request->input('module'); // 'leave' | 'reimbursement' | 'travel' | dll
        $action  = $request->input('action'); // 'approved' | 'rejected' | 'first level approval' | dll

        $user = Auth::user();

        // Hitung unread
        $unreadCount = $user->unreadNotifications()->count();

        // Query notifications (latest first)
        $query = $user->notifications()->latest();

        // Filter opsional berdasarkan payload data
        if ($module) {
            $query->whereJsonContains('data->module', $module);
        }
        if ($action) {
            $query->whereJsonContains('data->action', $action);
        }

        $paginator = $query->paginate($perPage, ['*'], 'page', $page);

        // Transform tiap item agar FE rapi & aman null
        $items = collect($paginator->items())->map(function ($n) {
            $data = $this->normalizeData($n->data);

            // Fallback urutan: title|subject|message|body|data
            $title = $data['title']
                ?? $data['subject']
                ?? $data['message']
                ?? $data['body']
                ?? $data['data']     // ← payload kamu selama ini
                ?? null;

            $body = $data['body']
                ?? $data['message']
                ?? $data['data']
                ?? null;

            return [
                'id'         => $n->id,
                'title'      => $title,
                'body'       => $body,
                'module'     => $data['module'] ?? null,
                'action'     => $data['action'] ?? null,
                'entity_id'  => $data['id'] ?? null,
                'by'         => $data['by'] ?? null,
                'read_at'    => optional($n->read_at)->toIso8601String(),
                'created_at' => optional($n->created_at)->toIso8601String(),
            ];
        })->values();

        // Bentuk respons: unread + objek paginator seperti yang dipakai FE
        return response()->json([
            'unread' => $unreadCount,
            'notifications' => [
                'current_page' => $paginator->currentPage(),
                'last_page'    => $paginator->lastPage(),
                'per_page'     => $paginator->perPage(),
                'total'        => $paginator->total(),
                'data'         => $items,
            ],
        ], 200);
    }

    /**
     * GET /api/notifications/{id}
     * Ambil satu notifikasi (untuk detail/preview).
     */
    public function show(string $id)
    {
        $user = Auth::user();
        $n = $user->notifications()->where('id', $id)->firstOrFail();

        $data = $this->normalizeData($n->data);

        $title = $data['title']
            ?? $data['subject']
            ?? $data['message']
            ?? $data['body']
            ?? $data['data']
            ?? null;

        $body = $data['body']
            ?? $data['message']
            ?? $data['data']
            ?? null;

        return response()->json([
            'id'         => $n->id,
            'title'      => $title,
            'body'       => $body,
            'module'     => $data['module'] ?? null,
            'action'     => $data['action'] ?? null,
            'entity_id'  => $data['id'] ?? null,
            'by'         => $data['by'] ?? null,
            'read_at'    => optional($n->read_at)->toIso8601String(),
            'created_at' => optional($n->created_at)->toIso8601String(),
        ], 200);
    }

    /**
     * POST /api/notifications/mark-as-read
     * Body:
     *  - id: string (opsional). Jika diisi → mark 1 notifikasi. Jika kosong → mark all.
     */
    public function markAsRead(Request $request)
    {
        $user = Auth::user();
        $id   = $request->input('id');

        if ($id) {
            $n = $user->unreadNotifications()->where('id', $id)->first();
            if ($n) {
                $n->markAsRead();
            }
        } else {
            $user->unreadNotifications->markAsRead();
        }

        return response()->json(['success' => true], 200);
    }

    /**
     * DELETE /api/notifications/{id}
     * (opsional) Hapus 1 notifikasi.
     */
    public function destroy(string $id)
    {
        $user = Auth::user();
        $n = $user->notifications()->where('id', $id)->firstOrFail();
        $n->delete();

        return response()->json(['success' => true], 200);
    }

    // ========================= Helpers =========================

    /**
     * Normalisasi payload data notifikasi.
     * - Jika array → kembalikan apa adanya
     * - Jika string JSON → decode
     * - Selain itu → array kosong
     */
    private function normalizeData($raw): array
    {
        if (is_array($raw)) {
            return $raw;
        }
        if (is_string($raw)) {
            $decoded = json_decode($raw, true);
            return is_array($decoded) ? $decoded : [];
        }
        return [];
    }
}
