<?php

namespace App\Http\Controllers\API;

use Auth;
use App\company;
use App\department;
use App\Employee;
use App\Reimbursement;
use App\ReimbursementType;
use App\User;
use App\Http\Controllers\Controller;
use App\Notifications\ReimburseNotify;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Notifications\GeneralNotificationtoAdmin;
use Response;


class ReimbursementController extends Controller
{
    public function index() {
        $reimbursement = Reimbursement::where('employee_id', Auth::user()->id)->get();
		return Response::json($reimbursement, 200);
    }

    public function list_approval(Request $request)
    {
        // Get the currently authenticated user
        $logged_user = auth()->user();
        $employee = Employee::find($logged_user->id);
        
        if($employee->supervisor == 0 && $employee->dept_head == 0)
        {
            $paginatedLeaves = array();
            return Response::json($paginatedLeaves, 200);
        }
        
        // Default pagination values
        $perPage = $request->input('limit', 10); // Default to 10 items per page
        $page = $request->input('page', 1); // Default to page 1

        // Start building the query with relationships
        $reimbursement = Reimbursement::with('employee', 'employee.company', 'reimbursementType')
        ->orderByRaw("CASE WHEN status = 'Pending' THEN 0 ELSE 1 END")
        ->orderBy('created_at', 'ASC');

        // Supervisor-specific logic
        if (isset($employee->supervisor) && $employee->supervisor == 1) {
            $reimbursement = $reimbursement->whereHas('employee', function($query) use ($employee) {
                $query->where('dept_head', 0)->where('supervisor', 0)->where('coo', 0)
                    ->where('department_id', $employee->department_id);
            });
        }

        // Dept Head-specific logic
        if (isset($employee->dept_head) && $employee->dept_head == 1) {
            $reimbursement = $reimbursement->whereHas('employee', function($query) use ($employee) {
                $query->where('dept_head', 0)->where('coo', 0)
                    ->where('department_id', $employee->department_id);
            });
        }

        // COO-specific logic
        if (isset($employee->coo) && $employee->coo == 1) {
            $reimbursement = $reimbursement->whereHas('employee', function($query) use ($employee) {
                $query->where('coo', 0)
                    ->where('company_id', $employee->company_id);
            });
        }

        // Apply pagination
        $paginatedReimbursements = $reimbursement->paginate($perPage, ['*'], 'page', $page);

        // Return the paginated response as JSON
        return Response::json($paginatedReimbursements, 200);
    }
    
    public function update(Request $request)
    {
        $logged_user = auth()->user();

        if ($logged_user->can('edit-reimbursement')) {
            $id = $request->id;

            $validator = Validator::make(
                $request->only(
                    'id',
                    'employee_id',
                ),
                [
                    'id' => 'required',
                    'employee_id' => 'required',
                ]
            );


            if ($validator->fails()) {
                return response()->json(['errors' => $validator->errors()->all()]);
            }

            $current_employee = Employee::find($logged_user->id);
			$requester_employee = Employee::find($request->employee_id);
            $checking_reimburse = Reimbursement::find($id);
            if(!$logged_user->hasRole('admin')) {
                if($current_employee->dept_head == 1 && $checking_reimburse->status != "first level approval" && $requester_employee->supervisor != 1)
                {
                    return response()->json(['limit' => __('Manager allowed to approved first level approval and supervisor request')]);
                }
                elseif($current_employee->supervisor == 1 && ($checking_leave->status != "pending" || $checking_leave->status == "first level approval"))
				{
				    return response()->json(['error' => __('Supervisor only allowed to first level approval')]);
				}
            }

            $data = [];
            $data['employee_id'] = $request->employee_id;
            $data['status'] = $request->status ?? 'pending';

            Reimbursement::find($id)->update($data);
            
            if ($data['status'] != 'pending') {
                $text = "Your reimbursement status has been ". $data['status'];
				$notifiable = User::findOrFail($data['employee_id']);
				$notifiable->notify(new ReimburseNotify($text));
			}

            return response()->json(['success' => __('Data is successfully updated')]);
        } else {

            return response()->json(['error' => __('You are not authorized')]);
        }
    }

    public function approve(Request $request)
    {
        $request->validate([
            'id'     => 'required|integer|exists:reimbursements,id',
            'status' => 'nullable|string|in:approved,first level approval,pending',
            'note'   => 'nullable|string',
        ]);

        $reimb = Reimbursement::findOrFail($request->id);
        $prev  = trim((string) $reimb->status);
        $new   = $request->input('status', 'approved');

        $reimb->update(['status' => $new]);

        // === Kirim notifikasi ke 3 pihak ===
        $this->notifyReimb($reimb, $new, $request->input('note'));

        return response()->json([
            'success' => true,
            'message' => 'Reimbursement status updated',
            'data'    => ['id' => $reimb->id, 'previous_status' => $prev, 'new_status' => $new],
        ]);
    }

    public function reject(Request $request)
    {
        $request->validate([
            'id'     => 'required|integer|exists:reimbursements,id',
            'status' => 'nullable|string|in:rejected',
            'note'   => 'nullable|string',
        ]);

        $reimb = Reimbursement::findOrFail($request->id);
        $prev  = trim((string) $reimb->status);
        $new   = $request->input('status', 'rejected');

        $reimb->update(['status' => $new]);

        $this->notifyReimb($reimb, $new, $request->input('note'));

        return response()->json([
            'success' => true,
            'message' => 'Reimbursement status updated',
            'data'    => ['id' => $reimb->id, 'previous_status' => $prev, 'new_status' => $new],
        ]);
    }

    private function notifyReimb(Reimbursement $reimb, string $status, ?string $note = null): void
    {
        $entityId  = $reimb->id;
        $requester = \App\Employee::find($reimb->employee_id);
        if (!$requester) return;

        // Pesan dan link (sesuaikan link sesuai kebutuhanmu)
        $msgUser   = "Your reimbursement status is --- {$status}" . ($note ? " ({$note})" : "");
        $linkUser  = "/profile#Employee_reimbursement";

        $msgAdmin  = "Reimbursement #{$entityId} ({$requester->first_name} {$requester->last_name}) is {$status}";
        $linkAdmin = "https://hr-timas.rekret.com/requests/reimburse";

        // 1) Pemohon
        if ($user = User::find($reimb->employee_id)) {
            $user->notify(new GeneralNotificationtoAdmin($msgUser, $linkUser));
        }

        // 2) Dept Head & Supervisor
        $dept = department::find($requester->department_id);
        if (!empty($dept?->department_head)) {
            if ($lead = User::find($dept->department_head)) {
                $lead->notify(new GeneralNotificationtoAdmin($msgAdmin, $linkAdmin));
            }
        }
        if (!empty($dept?->supervisor)) {
            if ($spv = User::find($dept->supervisor)) {
                $spv->notify(new GeneralNotificationtoAdmin($msgAdmin, $linkAdmin));
            }
        }

        // 3) HR (role_users_id = 6) di company yang sama
        $hrs = User::where('role_users_id', 6)
            ->whereJsonContains('company_ids', (int) $requester->company_id)
            ->get();

        foreach ($hrs as $hr) {
            $hr->notify(new GeneralNotificationtoAdmin($msgAdmin, $linkAdmin));
        }
    }

}
