<?php

namespace App\Http\Controllers\API;

use Auth;
use App\company;
use App\department;
use App\Employee;
use App\Travel;
use App\TravelType;
use App\User;
use App\Http\Controllers\Controller;
use App\Notifications\EmployeeTravelStatus;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class TravelController extends Controller
{
    /**
     * List travel milik user login (non-paginated).
     * (Dipakai bila kamu butuh daftar cepat tanpa pagination)
     */
    public function index()
    {
        $travel = Travel::with('employee', 'employee.company', 'travelType')
            ->where('employee_id', Auth::user()->id)
            ->orderBy('id', 'DESC')
            ->get();

        return Response::json($travel, 200);
    }

    /**
     * List untuk tab APPROVAL.
     * Mengikuti pola ReimbursementController: paginate default Laravel.
     * FE kamu sudah terbukti bisa konsumsi bentuk paginator ini.
     */
    public function list_approval(Request $request)
    {
        $logged_user = auth()->user();
        $employee    = Employee::find($logged_user->id);

        // Jika bukan SPV/DeptHead/COO, tidak ada daftar approval
        if (
            empty($employee) ||
            ((int)$employee->supervisor === 0 && (int)$employee->dept_head === 0 && (int)$employee->coo === 0)
        ) {
            // Samakan format seperti reimbursement: kembalikan array kosong
            return Response::json([], 200);
        }

        $perPage = $request->input('limit', 10);
        $page    = $request->input('page', 1);

        $travel = Travel::with('employee', 'employee.company', 'travelType')
            // Urutkan 'Pending' (atau 'pending') ke atas lalu berdasarkan waktu lama ke baru,
            // sama seperti reimbursement kamu.
            ->orderByRaw("CASE WHEN LOWER(status) = 'pending' THEN 0 ELSE 1 END")
            ->orderBy('created_at', 'ASC');

        // Supervisor: bawahan dalam departemen yang bukan spv/dept_head/coo
        if ((int)$employee->supervisor === 1) {
            $travel = $travel->whereHas('employee', function ($q) use ($employee) {
                $q->where('department_id', $employee->department_id)
                  ->where('supervisor', 0)
                  ->where('dept_head', 0)
                  ->where('coo', 0);
            });
        }

        // Dept Head: semua di departemen (kecuali dept_head/coo)
        if ((int)$employee->dept_head === 1) {
            $travel = $travel->whereHas('employee', function ($q) use ($employee) {
                $q->where('department_id', $employee->department_id)
                  ->where('dept_head', 0)
                  ->where('coo', 0);
            });
        }

        // COO: semua di company yang sama (kecuali COO)
        if ((int)$employee->coo === 1) {
            $travel = $travel->whereHas('employee', function ($q) use ($employee) {
                $q->where('company_id', $employee->company_id)
                  ->where('coo', 0);
            });
        }

        $paginatedTravel = $travel->paginate($perPage, ['*'], 'page', $page);

        return Response::json($paginatedTravel, 200);
    }

    /**
     * Update status travel (dipanggil di /travels/approval).
     * Diselaraskan dengan pola reimbursement::update namun memperbaiki bug lama:
     * - Hilangkan variabel tak terdefinisi ($checking_leave)
     * - Validasi minimal id & employee_id
     */
    public function update(Request $request)
    {
        $logged_user = auth()->user();

        if (!$logged_user->can('edit-travel')) {
            return response()->json(['error' => __('You are not authorized')], 403);
        }

        $validator = Validator::make(
            $request->only('id', 'employee_id', 'status'),
            [
                'id'          => 'required|integer|exists:travels,id',
                'employee_id' => 'required|integer|exists:employees,id',
                'status'      => 'nullable|string|in:pending,first level approval,approved,rejected',
            ]
        );

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()->all()], 422);
        }

        $id                = (int) $request->id;
        $employeeId        = (int) $request->employee_id;
        $current_employee  = Employee::find($logged_user->id);
        $requesterEmployee = Employee::find($employeeId);
        $checking_travel   = Travel::find($id);

        // Batasi sesuai aturanmu (adaptasi dari reimbursement)
        if (!$logged_user->hasRole('admin')) {
            // Manager (dept_head=1) hanya boleh approve first level + supervisor request
            if (
                (int)$current_employee->dept_head === 1 &&
                strtolower($checking_travel->status ?? 'pending') !== 'first level approval' &&
                (int)$requesterEmployee->supervisor !== 1
            ) {
                return response()->json([
                    'limit' => __('Manager allowed to approved first level approval and supervisor request')
                ], 422);
            }

            // Supervisor hanya boleh first level dari pending
            if (
                (int)$current_employee->supervisor === 1 &&
                !in_array(strtolower($checking_travel->status ?? 'pending'), ['pending', 'first level approval'], true)
            ) {
                return response()->json(['error' => __('Supervisor only allowed to first level approval')], 422);
            }
        }

        // Tentukan status baru (fallback ke status lama bila tidak dikirim)
        $newStatus = $request->status ?? $checking_travel->status ?? 'pending';

        $data = [
            'employee_id' => $employeeId,
            'status'      => $newStatus,
        ];

        $checking_travel->update($data);

        // Notifikasi bila status bukan pending
        if (strtolower($data['status']) !== 'pending') {
            $notifiable = User::findOrFail($data['employee_id']);
            $notifiable->notify(new EmployeeTravelStatus($data['status']));
        }

        return response()->json(['success' => __('Data is successfully updated')]);
    }

    /**
     * Versi approve eksplisit sederhana (opsional).
     * Kalau ingin sangat mirip reimbursement::approve, kamu bisa panggil update dengan status dari request.
     */
    public function approve(Request $request)
    {
        $request->validate([
            'id'     => 'required|integer|exists:travels,id',
            'status' => 'nullable|string|in:approved,first level approval,pending',
        ]);

        $travel = Travel::findOrFail($request->id);
        $prev   = trim((string) $travel->status);
        $new    = $request->input('status', 'approved');

        $travel->update(['status' => $new]);

        // Notifikasi ke pemohon
        try {
            if ($user = User::find($travel->employee_id)) {
                $user->notify(new EmployeeTravelStatus($new));
            }
        } catch (\Throwable $e) {
            // ignore notif failure
        }

        return response()->json([
            'success' => true,
            'message' => 'Travel status updated',
            'data'    => ['id' => $travel->id, 'previous_status' => $prev, 'new_status' => $new],
        ]);
    }

    /**
     * Master Travel Type
     */
    public function travel_type()
    {
        $types = TravelType::orderBy('id')->get();
        return response()->json(['data' => $types, 'success' => true], 200);
    }

    /**
     * Master Travel Mode
     */
    public function travel_mode()
    {
        $mode = ["By Bus", "By Train", "By Plane", "By Taxi", "By Rental Car", "By Other"];
        return response()->json(['data' => $mode, 'success' => true], 200);
    }
}
