<?php

namespace App\Http\Controllers;

use App\company;
use App\location;
use App\Employee;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Validator;

class CompanyController extends Controller {

	/**
	 * Display a listing of the resource.
	 *
	 * @return \Illuminate\Http\Response
	 */
	public function index()
	{
		$locations = location::all('id','location_name');
		if (request()->ajax())
		{
			return datatables()->of(company::with('Location.Country')->latest()->get())
				->setRowId(function ($company)
				{
					return $company->id;
				})
				->addColumn('city',function ($row)
				{
					return optional($row->Location)->city ?? '';
				})
				->addColumn('country',function ($row)
				{
					return optional(optional($row->Location)->Country)->name ?? '';
				})
				->addColumn('coo',function ($row)
				{
					if ($row->coo_name) {
						return $row->coo_name->first_name. ' ' . $row->coo_name->last_name;
					} else {
						return '';
					}
				})
				->addColumn('action', function ($data)
				{
					$button = '<button type="button" name="show" id="' . $data->id . '" class="show_new btn btn-success btn-sm"><i class="dripicons-preview"></i></button>';
					$button .= '&nbsp;&nbsp;';
					if(auth()->user()->can('edit-company'))
					{
						$button .= '<button type="button" name="edit" id="' . $data->id . '" class="edit btn btn-primary btn-sm"><i class="dripicons-pencil"></i></button>';
						$button .= '&nbsp;&nbsp;';
					}
					if(auth()->user()->can('delete-company'))
					{
						$button .= '<button type="button" name="delete" id="' . $data->id . '" class="delete btn btn-danger btn-sm"><i class="dripicons-trash"></i></button>';
					}
					return $button;
				})
				->rawColumns(['action'])
				->make(true);
		}

		return view('organization.company.index',compact('locations'));
	}


	/**
	 * Store a newly created resource in storage.
	 *
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\Response
	 */

	public function store(Request $request)
	{
		if(auth()->user()->can('store-company'))
		{
			$validator = Validator::make($request->only('company_name', 'company_type', 'trading_name', 'registration_no', 'contact_no', 'email', 'website', 'tax_no',
				'location_id', 'company_logo', 'start_cut_off_date', 'end_cut_off_date', 'leave_amount', 'long', 'lat', 'radius'),
				[
					'company_name' => 'required|unique:companies,company_name,',
					'company_type' => 'required',
					'email' => 'required|email',
					'contact_no' => 'required|numeric',
					// 'location_id' => 'nullable',
					'company_logo' => 'nullable|image|max:2048|mimes:jpeg,png,jpg,gif',
					'start_cut_off_date' => 'nullable',
					'end_cut_off_date' => 'nullable',
					'leave_amount' => 'nullable',
					// 'long' => 'required',
					// 'lat' => 'required',
					// 'radius' => 'required|numeric'
				]
			);


			if ($validator->fails())
			{
				return response()->json(['errors' => $validator->errors()->all()]);
			}

			$data = [];

			$data['company_name'] = $request->company_name;
			$data['company_type'] = $request->company_type;
			$data ['trading_name'] = $request->trading_name;
			$data ['registration_no'] = $request->registration_no;
			$data ['contact_no'] = $request->contact_no;
			$data ['email'] = $request->email;
			$data ['website'] = $request->website;
			$data ['tax_no'] = $request->tax_no;
			$data ['location_id'] = $request->location_id ?: null;
			$data ['start_cut_off_date'] = $request->start_cut_off_date;
			$data ['end_cut_off_date'] = $request->end_cut_off_date;
			$data ['leave_amount'] = $request->leave_amount;
			$data ['long'] = $request->long;
			$data ['lat'] = $request->lat;
			$data ['radius'] = $request->radius;

			$company_logo = $request->company_logo;

			if (isset($company_logo))
			{

				if ($company_logo->isValid())
				{
					$file_name = preg_replace('/\s+/', '', rand()) . '_' . time() . '.' . $company_logo->getClientOriginalExtension();
					$company_logo->storeAs('company_logo', $file_name);
					$data['company_logo'] = $file_name;
				}
			}

			if($request->employee_id)
			{
				$data ['coo'] = $request->employee_id;
				$up_dept['coo'] = 1;
			}
			else{
				$data ['coo'] = NULL;
				$up_dept['coo'] = 0;
			}

			if(!empty($request->employee_code))
			{
				$data ['employee_code'] = $request->employee_code;
			}

			company::create($data);


			return response()->json(['success' => __('Data Added successfully.')]);
		}
		return response()->json(['success' => __('You are not authorized')]);
	}


	/**
	 * Display the specified resource.
	 *
	 * @param int $id
	 * @return \Illuminate\Http\Response
	 */
	public function show($id)
	{
		if (request()->ajax())
		{
			$data = company::with('location.country')->findOrFail($id);

			return response()->json(['data' => $data]);
		}
	}

	/**
	 * Show the form for editing the specified resource.
	 *
	 * @param int $id
	 * @return \Illuminate\Http\Response
	 */
	public function edit($id)
	{

		if (request()->ajax())
		{
			$data = company::findOrFail($id);
			$employees = Employee::select('id', 'first_name', 'last_name')->where('company_id', $id)->where('is_active',1)
            ->where('exit_date',NULL)->get();

			return response()->json(['data' => $data, 'employees' => $employees]);
		}
	}

	/**
	 * Update the specified resource in storage.
	 *
	 * @param \Illuminate\Http\Request $request
	 * @param int $id
	 * @return \Illuminate\Http\Response
	 */
	public function update(Request $request)
	{

		$logged_user = auth()->user();

		if ($logged_user->can('edit-company'))
		{
			$id = $request->hidden_id;

			$validator = Validator::make($request->only('company_name', 'company_type', 'trading_name', 'registration_no', 'contact_no', 'email', 'website', 'tax_no',
				'location_id', 'company_logo', 'start_cut_off_date', 'end_cut_off_date', 'leave_amount', 'long', 'lat', 'radius'),
				[
					'company_name' => 'required|unique:companies,company_name,' . $id,
					'company_type' => 'required',
					'email' => 'required|email',
					'contact_no' => 'required|numeric',
					// other fields optional
					// 'location_id' => 'nullable',
					// 'company_logo' => 'nullable|image|max:2048|mimes:jpeg,png,jpg,gif',
					// 'start_cut_off_date' => 'nullable',
					// 'end_cut_off_date' => 'nullable',
					// 'leave_amount' => 'nullable',
				]
			);


			if ($validator->fails())
			{
				return response()->json(['errors' => $validator->errors()->all()]);
			}


			$data = [];

			$data['company_name'] = $request->company_name;
			$data ['trading_name'] = $request->trading_name;
			$data ['registration_no'] = $request->registration_no;
			$data ['contact_no'] = $request->contact_no;
			$data ['email'] = $request->email;
			$data ['website'] = $request->website;
			$data ['tax_no'] = $request->tax_no;
			$data ['location_id'] = $request->location_id ?: null;
			$data ['start_cut_off_date'] = $request->start_cut_off_date;
			$data ['end_cut_off_date'] = $request->end_cut_off_date;
			$data ['leave_amount'] = $request->leave_amount;
			$data ['long'] = $request->long;
			$data ['lat'] = $request->lat;
			$data ['radius'] = $request->radius;

			if ($request->company_type)
			{
				$data ['company_type'] = $request->company_type;
			}


			$company_logo = $request->company_logo;

			if (isset($company_logo))
			{

				if ($company_logo->isValid())
				{
					$file_name = preg_replace('/\s+/', '', rand()) . '_' . time() . '.' . $company_logo->getClientOriginalExtension();
					$company_logo->storeAs('company_logo', $file_name);
					$data['company_logo'] = $file_name;
				}
			}

			if($request->employee_id)
			{
				$data ['coo'] = $request->employee_id;
				$up_dept['coo'] = 1;
			}
			else{
				$data ['coo'] = NULL;
				$up_dept['coo'] = 0;
			}

			if(!empty($request->employee_code))
			{
				$data ['employee_code'] = $request->employee_code;
			}

			$dept_head = Employee::whereId($request->employee_id)->update($up_dept);

			company::whereId($id)->update($data);

			return response()->json(['success' => __('Data is successfully updated')]);

		} else
		{
			return response()->json(['success' => __('You are not authorized')]);
		}


	}

	/**
	 * Remove the specified resource from storage.
	 *
	 * @param int $id
	 * @return \Illuminate\Http\Response
	 */
	public function destroy($id)
	{
		if(!env('USER_VERIFIED'))
		{
			return response()->json(['error' => 'This feature is disabled for demo!']);
		}
		$logged_user = auth()->user();

		if ($logged_user->can('delete-company'))
		{
			company::whereId($id)->delete();

			return response()->json(['success' => __('Data is successfully deleted')]);

		}

		return response()->json(['success' => __('You are not authorized')]);

	}


	public function delete_by_selection(Request $request)
	{
		if(!env('USER_VERIFIED'))
		{
			return response()->json(['error' => 'This feature is disabled for demo!']);
		}
		$logged_user = auth()->user();

		if ($logged_user->can('delete-company'))
		{

			$company_id = $request['companyIdArray'];
			$company = company::whereIntegerInRaw('id', $company_id);

			if ($company->delete())
			{
				return response()->json(['success' => __('Multi Delete',['key'=>trans('file.Company')])]);
			} else
			{
				return response()->json(['error' => 'Error,selected users can not be deleted']);
			}
		}
		return response()->json(['success' => __('You are not authorized')]);
	}

}


