<?php

namespace App\Http\Controllers;

use App\Tax;
use App\company;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class TaxController extends Controller
{
    public function index()
    {
        $logged_user = auth()->user();
        $companies = company::select('id', 'company_name')->get();

        if ($logged_user->can('view-tax')) {
            if (request()->ajax()) {
                return datatables()->of(Tax::with('company')->get())
                    ->setRowId(function ($tax) {
                        return $tax->id;
                    })
                    ->addColumn('company_name', function ($row) {
                        return $row->company->company_name;
                    })
                    ->addColumn('min_salary', function ($row) {
                        return $row->min_salary_formatted;
                    })
                    ->addColumn('max_salary', function ($row) {
                        return $row->max_salary_formatted;
                    })
                    ->addColumn('percentage', function ($row) {
                        return $row->percentage . ' %';
                    })
                    ->addColumn('action', function ($data) {
                        $button = '<button type="button" name="show" id="' . $data->id . '" class="show_new btn btn-success btn-sm"><i class="dripicons-preview"></i></button>';
                        $button .= '&nbsp;&nbsp;';
                        if (auth()->user()->can('edit-tax')) {
                            $button .= '<button type="button" name="edit" id="' . $data->id . '" class="edit btn btn-primary btn-sm"><i class="dripicons-pencil"></i></button>';
                            $button .= '&nbsp;&nbsp;';
                        }
                        if (auth()->user()->can('delete-tax')) {
                            $button .= '<button type="button" name="delete" id="' . $data->id . '" class="delete btn btn-danger btn-sm"><i class="dripicons-trash"></i></button>';
                        }

                        return $button;
                    })
                    ->rawColumns(['action'])
                    ->make(true);
            }

            return view('core_hr.tax.index', compact('companies'));
        }

        return abort('403', __('You are not authorized'));
    }

    public function store(Request $request)
    {
        $logged_user = auth()->user();

        if ($logged_user->can('store-tax')) {
            $validator = Validator::make(
                $request->only(
                    'company_id',
                    'min_salary',
                    'max_salary',
                    'percentage'
                ),
                [
                    'company_id' => 'required',
                    'min_salary' => 'required',
                    'max_salary' => 'required',
                    'percentage' => 'required',
                ]
            );


            if ($validator->fails()) {
                return response()->json(['errors' => $validator->errors()->all()]);
            }


            $data = [];

            $data['company_id'] = $request->company_id;
            $data['min_salary'] = $request->min_salary;
            $data['max_salary'] = $request->max_salary;
            $data['percentage'] = $request->percentage;
            Tax::create($data);

            return response()->json(['success' => __('Data Added successfully.')]);
        }

        return response()->json(['success' => __('You are not authorized')]);
    }

    public function edit($id)
    {
        if (request()->ajax()) {
            $data = Tax::with('company')->findOrFail($id);

            return response()->json(['data' => $data]);
        }
    }

    public function update(Request $request)
    {
        $logged_user = auth()->user();

        if ($logged_user->can('edit-tax')) {
            $id = $request->hidden_id;

            $validator = Validator::make(
                $request->only(
                    'company_id',
                    'min_salary',
                    'max_salary',
                    'percentage'
                ),
                [
                    'company_id' => 'required',
                    'min_salary' => 'required',
                    'max_salary' => 'required',
                    'percentage' => 'required',
                ]
            );


            if ($validator->fails()) {
                return response()->json(['errors' => $validator->errors()->all()]);
            }


            $data = [];

            $data['company_id'] = $request->company_id;
            $data['min_salary'] = $request->min_salary;
            $data['max_salary'] = $request->max_salary;
            $data['percentage'] = $request->percentage;

            Tax::find($id)->update($data);

            return response()->json(['success' => __('Data is successfully updated')]);
        }
        return response()->json(['success' => __('You are not authorized')]);
    }

    public function destroy($id)
    {
        if (!env('USER_VERIFIED')) {
            return response()->json(['error' => 'This feature is disabled for demo!']);
        }
        $logged_user = auth()->user();

        if ($logged_user->can('delete-tax')) {
            Tax::whereId($id)->delete();

            return response()->json(['success' => __('Data is successfully deleted')]);
        }
        return response()->json(['success' => __('You are not authorized')]);
    }
    public function delete_by_selection(Request $request)
    {
        if (!env('USER_VERIFIED')) {
            return response()->json(['error' => 'This feature is disabled for demo!']);
        }
        $logged_user = auth()->user();

        if ($logged_user->can('delete-tax')) {

            $tax_id = $request['taxIdArray'];
            $tax = Tax::whereIntegerInRaw('id', $tax_id);
            if ($tax->delete()) {
                return response()->json(['success' => __('Multi Delete', ['key' => __('BPJS TK')])]);
            } else {
                return response()->json(['error' => 'Error, selected BPJS TK can not be deleted']);
            }
        }
        return response()->json(['success' => __('You are not authorized')]);
    }
    public function show($id)
    {
        if (request()->ajax()) {
            $data = Tax::with('company')->findOrFail($id);
            return response()->json(['data' => $data]);
        }
    }
}
