<?php

namespace App\Http\Controllers;

use App\company;
use App\Employee;
use App\Http\Controllers\Controller;
use App\Reimbursement;
use App\ReimbursementType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ReimbursementController extends Controller
{
    public function index()
    {
        $logged_user = auth()->user();
        $companyId = null;
		if ($logged_user->role_users_id == 6) {
			$companyId = json_decode($logged_user->company_ids);
		}
		else
		{
			$companyId[] = $logged_user->company_ids;
		}

		if ($logged_user->role_users_id == 1)
        {
            $companies = company::select('id', 'company_name')->get();
        }
        else
        {
            $companies = company::select('id', 'company_name')->whereIn('id', $companyId)->get();   
        }
        
        $reimbursement_types = ReimbursementType::select('id', 'name')->get();

        if ($logged_user->can('view-reimbursement')) {
            if (request()->ajax()) {
                $reimbursements = Reimbursement::with(['employee', 'employee.company', 'reimbursementType']);
                if ($companyId[0]) {
                    $reimbursements = $reimbursements->whereHas('employee', function ($query) use ($companyId) {
                        return $query->whereIn('company_id', $companyId);
                    });
                }
                return datatables()->of($reimbursements->get())
                    ->setRowId(function ($row) {
                        return $row->id;
                    })
                    ->addColumn('reimbursement_type', function ($row) {
                        return $row->reimbursementType->name ?? '';
                    })
                    ->addColumn('company', function ($row) {
                        return $row->employee->company->company_name ?? '';
                    })
                    ->addColumn('employee', function ($row) {
                        return $row->employee->full_name ?? '';
                    })
                    ->addColumn('amount', function ($row) {
                        return $row->amount_formatted;
                    })
                    ->addColumn('status', function ($row) {
                        return "<div class='badge badge-success'>" . $row->status . "</div>";
                    })
                    ->addColumn('action', function ($data) {
                        $button = '<button type="button" name="show" id="' . $data->id . '" class="show_reimbursement btn btn-success btn-sm"><i class="dripicons-preview"></i></button>';
                        $button .= '&nbsp;&nbsp;';
                        if (auth()->user()->can('edit-reimbursement')) {
                            $button .= '<button type="button" name="edit" id="' . $data->id . '" class="edit btn btn-primary btn-sm"><i class="dripicons-pencil"></i></button>';
                            $button .= '&nbsp;&nbsp;';
                        }
                        if (auth()->user()->can('delete-reimbursement')) {
                            $button .= '<button type="button" name="delete" id="' . $data->id . '" class="delete btn btn-danger btn-sm"><i class="dripicons-trash"></i></button>';
                        }

                        return $button;
                        return $button;
                    })
                    ->rawColumns(['action', 'status'])
                    ->make(true);
            }

            return view('core_hr.reimbursement.index', compact('companies', 'reimbursement_types'));
        }

        return abort('403', __('You are not authorized'));
    }

    public function store(Request $request)
    {
        if (auth()->user()->can('store-reimbursement') || auth()->user()) {
            $validator = Validator::make(
                $request->only(
                    'employee_id',
                    'reimbursement_type_id',
                    'amount',
                    'description',
                    'reimbursement_attachments',
                ),
                [
                    'employee_id' => 'required',
                    'reimbursement_type_id' => 'required',
                    'amount' => 'required',
                    'description' => 'required',
                    'reimbursement_attachments' => 'nullable|file|max:10240|mimes:jpeg,png,jpg,gif,ppt,pptx,doc,docx,pdf',
                ]
            );


            if ($validator->fails()) {
                return response()->json(['errors' => $validator->errors()->all()]);
            }


            $data = [];

            $data['employee_id'] = $request->employee_id;
            $data['reimbursement_type_id'] = $request->reimbursement_type_id;
            $data['amount'] = $request->amount;
            $data['description'] = $request->description;
            $data['status'] = $request->status ?? 'pending';

            $file = $request->reimbursement_attachments;

            $file_name = null;

            if (isset($file)) {
                if ($file->isValid()) {
                    $file_name = 'reimbursement_' . time() . '.' . $file->getClientOriginalExtension();
                    $file->storeAs('reimbursement_attachments', $file_name);
                    $data['attachment'] = $file_name;
                }
            }
            Reimbursement::create($data);

            return response()->json(['success' => __('Data Added successfully.')]);
        }

        return response()->json(['success' => __('You are not authorized')]);
    }

    public function show($id)
    {
        if (request()->ajax()) {
            $data = Reimbursement::with(['reimbursementType', 'employee', 'employee.company'])->findOrFail($id);

            return response()->json(['data' => $data]);
        }
    }

    public function edit($id)
    {
        if (request()->ajax()) {
            $data = Reimbursement::with(['reimbursementType', 'employee', 'employee.company'])->findOrFail($id);

            $employees = Employee::select('id', 'first_name', 'last_name')
                ->where('company_id', $data->employee->company_id)->where('is_active', 1)->where('exit_date', NULL)->get();

            return response()->json(['data' => $data, 'employees' => $employees]);
        }
    }

    public function update(Request $request)
    {
        $logged_user = auth()->user();

        if (! $logged_user->can('edit-reimbursement')) {
            return response()->json(['success' => __('You are not authorized')]);
        }

        $id = $request->hidden_id;

        // Validasi input
        $validator = Validator::make(
            $request->only('employee_id', 'reimbursement_type_id', 'amount', 'description', 'status'),
            [
                'employee_id'           => 'required|integer',
                'reimbursement_type_id' => 'required|integer',
                'amount'                => 'required|numeric|min:0',
                'description'           => 'required|string',
                'status'                => 'nullable|in:pending,approved,rejected',
            ]
        );

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()->all()]);
        }

        // Ambil data yang akan diupdate
        $reimb = Reimbursement::findOrFail($id);

        // (Opsional) contoh rule role sederhana:
        // if (!$logged_user->hasRole('admin') && $logged_user->cannot('approve-reimbursement')) {
        //     return response()->json(['limit' => __('You are not allowed to approve this reimbursement')]);
        // }

        $data = [
            'employee_id'           => $request->employee_id,
            'reimbursement_type_id' => $request->reimbursement_type_id,
            'amount'                => $request->amount,
            'description'           => $request->description,
        ];

        // Hanya ubah status kalau dikirim dari form (approve/reject)
        if ($request->filled('status')) {
            $data['status'] = $request->status; // 'approved' | 'rejected' | 'pending'
        }

        $reimb->update($data);

        return response()->json(['success' => __('Data is successfully updated')]);
    }

    public function approve(Request $request)
    {
        $user = auth()->user();

        if (! $user->can('edit-reimbursement')) {
            return response()->json(['errors' => ['You are not authorized']], 403);
        }

        $data = $request->validate([
            'id' => 'required|integer|exists:reimbursements,id',
            // kalau nanti mau support reject:
            // 'action' => 'nullable|in:approve,reject'
        ]);

        $reimb   = Reimbursement::findOrFail($data['id']);
        $current = trim(strtolower($reimb->status)); // 'pending', 'first level approval', 'approved'
        $roleId  = (string) $user->role_users_id;

        $next = null;

        if ($roleId === '7') { // Supervisor
            if ($current === 'pending') {
                $next = 'first level approval';
            } else {
                return response()->json([
                    'errors' => ['Supervisor can only approve reimbursements that are pending.']
                ], 422);
            }
        } elseif ($roleId === '6') { // Manager
            if (in_array($current, ['pending', 'first level approval'], true)) {
                $next = 'approved';
            } else {
                return response()->json([
                    'errors' => ['Reimbursement already approved.']
                ], 422);
            }
        } else {
            return response()->json(['errors' => ['Your role is not allowed to approve']], 403);
        }

        $reimb->update(['status' => $next]);

        return response()->json([
            'success' => true,
            'message' => 'Status updated successfully',
            'data'    => [
                'id'               => $reimb->id,
                'previous_status'  => $current,
                'new_status'       => $next,
            ],
        ]);
    }

    public function destroy($id)
    {
        if (!env('USER_VERIFIED')) {
            return response()->json(['error' => 'This feature is disabled for demo!']);
        }
        $logged_user = auth()->user();

        if ($logged_user->can('delete-reimbursement')) {
            Reimbursement::whereId($id)->delete();

            return response()->json(['success' => __('Data is successfully deleted')]);
        }

        return response()->json(['success' => __('You are not authorized')]);
    }

    public function delete_by_selection(Request $request)
    {
        if (!env('USER_VERIFIED')) {
            return response()->json(['error' => 'This feature is disabled for demo!']);
        }
        $logged_user = auth()->user();

        if ($logged_user->can('delete-reimbursement')) {

            $reimbursement_id = $request['reimbursementIdArray'];
            $reimbursement = Reimbursement::whereIntegerInRaw('id', $reimbursement_id);
            if ($reimbursement->delete()) {
                return response()->json(['success' => 'Selected reimbursements has been deleted']);
            } else {
                return response()->json(['error' => 'Error, selected reimbursements can not be deleted']);
            }
        }

        return response()->json(['success' => __('You are not authorized')]);
    }
}
